{
    This file is part of RA1792Control

    RACAL RA1792 Control program

    Copyright (C) 2014-2022 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/.
}

unit ura1792;

{$mode objfpc}{$H+}

{$INCLUDE defines.inc}

interface

uses
  Classes, SysUtils, FileUtil, RTTICtrls, SpinEx, Forms, Controls, Graphics,
  Dialogs, StdCtrls, Menus, ExtCtrls, Buttons, ComCtrls, serial, types, LCLType,
  ActnList, Spin, ComboEx, mouse, dateutils, math, user, UAbout, UGNU, UMAN,
  urxaddr, umanager, uenttestno, JButton, lNetComponents, FileInfo,
  DefaultTranslator, LNET, uparse, utcp, uscan;

type

  { TRA1792 }

  TRA1792 = class(TForm)
    B0: TJButton;
    B1: TJButton;
    B2: TJButton;
    B3: TJButton;
    B4: TJButton;
    B5: TJButton;
    B6: TJButton;
    B7: TJButton;
    B8: TJButton;
    B9: TJButton;
    BAM: TJButton;
    BCHAN: TJButton;
    BCHSCAN: TJButton;
    BCW: TJButton;
    BENTER: TJButton;
    BFM: TJButton;
    BFREQ: TJButton;
    BISB: TJButton;
    BLSB: TJButton;
    BRCL: TJButton;
    BREM: TJButton;
    BSTORE: TJButton;
    BTUNE: TJButton;
    BBFO: TJButton;
    BUSB: TJButton;
    CBXBW: TComboBoxEx;
    CBXAGC: TComboBoxEx;
    CHANTENS: TLabel;
    CHANUNITS: TLabel;
    CBXMODE: TComboBoxEx;
    LBNETW: TLabel;
    MBSCANW: TMenuItem;
    Separator3: TMenuItem;
    Separator2: TMenuItem;
    MTCP: TMenuItem;
    RA1792Server: TLTCPComponent;
    MMAGALLF: TMenuItem;
    Separator1: TMenuItem;
    MPORTPROL: TMenuItem;
    MMAG100: TMenuItem;
    MMAG110: TMenuItem;
    MMAG125: TMenuItem;
    MMAG150: TMenuItem;
    MMAG175: TMenuItem;
    MMAG200: TMenuItem;
    N2: TMenuItem;
    MFONTMAGN: TMenuItem;
    N1: TMenuItem;
    MTIMED: TMenuItem;
    M0P5KHZT: TMenuItem;
    MSTBITE: TMenuItem;
    MSTBITEFM: TMenuItem;
    MOTHER: TMenuItem;
    RXFreq1H: TLabel;
    RXFreq1M: TLabel;
    RXFreq10H: TLabel;
    RXFreq100H: TLabel;
    RXFreq10M: TLabel;
    RXFreqDOT: TLabel;
    RXFreq1K: TLabel;
    RXFreq10K: TLabel;
    RXFreq100K: TLabel;
    LBAM: TLabel;
    LBBFO: TLabel;
    LBBFO1: TLabel;
    LBBFO2: TLabel;
    LBBFOFR: TLabel;
    LBBW: TLabel;
    LBBW1: TLabel;
    LBBW2: TLabel;
    LBCHAN: TLabel;
    LBCW: TLabel;
    LBDBU: TLabel;
    LBFAULT: TLabel;
    LBFM: TLabel;
    LBFREQ: TLabel;
    LBISB: TLabel;
    LBKHZ: TLabel;
    LBLONG: TLabel;
    LBLSB: TLabel;
    LBMAN: TLabel;
    LBMED: TLabel;
    LBMET: TLabel;
    LBMUTE: TLabel;
    SPFREQ: TSpinEditEx;
    STIND: TLabel;
    STREMOTE: TLabel;
    LBRF: TLabel;
    LBSCALE: TLabel;
    LBSCAN: TLabel;
    M10HZ: TMenuItem;
    M100HZ: TMenuItem;
    M1KHZ: TMenuItem;
    M1KHZT: TMenuItem;
    M10KHZT: TMenuItem;
    M100KHZT: TMenuItem;
    MenuItem4: TMenuItem;
    M1HZ: TMenuItem;
    MenuItem8: TMenuItem;
    LeftPanel: TPanel;
    RightPanel: TPanel;
    MenuItem2: TMenuItem;
    MenuItem3: TMenuItem;
    MRSTATE: TMenuItem;
    MMANCH: TMenuItem;
    MMANST: TMenuItem;
    MSCHANN: TMenuItem;
    MRXADD: TMenuItem;
    M9600B: TMenuItem;
    M38400B: TMenuItem;
    M57600B: TMenuItem;
    MSG: TMemo;
    MenuItem11: TMenuItem;
    MChannels: TMenuItem;
    MSHOWC: TMenuItem;
    MBlack: TMenuItem;
    MBlue: TMenuItem;
    MGreen: TMenuItem;
    MenuItem6: TMenuItem;
    MenuItem7: TMenuItem;
    MLEDC: TMenuItem;
    MenuItem9: TMenuItem;
    MENALL: TMenuItem;
    MGNULIC: TMenuItem;
    MABOUT: TMenuItem;
    MMAN: TMenuItem;
    MHelp: TMenuItem;
    MLSTATE: TMenuItem;
    MSSTATE: TMenuItem;
    MenuItem5: TMenuItem;
    MSmeter: TMenuItem;
    MenuItem1: TMenuItem;
    MCustom: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    SHKNOB1: TShape;
    SHKNOB: TShape;
    LBSHORT: TLabel;
    LBTUNE: TLabel;
    LBUSB: TLabel;
    TBATTEN: TTrackBar;
    UPDDISP: TTimer;
    TNET: TTimer;
    TSmeter: TTimer;
    TSMDISP: TTimer;
    MainMenu1: TMainMenu;
    MFile: TMenuItem;
    M19200B: TMenuItem;
    M115200B: TMenuItem;
    MExit: TMenuItem;
    MOptions: TMenuItem;
    MPort: TMenuItem;
    MSpeed: TMenuItem;
    MttyUSB0: TMenuItem;
    MttyUSB1: TMenuItem;
    procedure BCHANClick(Sender: TObject);
    procedure BCHSCANClick(Sender: TObject);
    procedure BCWClick(Sender: TObject);
    procedure BTUNEClick(Sender: TObject);
    procedure BENTERClick(Sender: TObject);
    procedure BSTOREClick(Sender: TObject);
    procedure BAMClick(Sender: TObject);
    procedure B4Click(Sender: TObject);
    procedure B7Click(Sender: TObject);
    procedure B8Click(Sender: TObject);
    procedure B9Click(Sender: TObject);
    procedure BFMClick(Sender: TObject);
    procedure BLSBClick(Sender: TObject);
    procedure B5Click(Sender: TObject);
    procedure B6Click(Sender: TObject);
    procedure B2Click(Sender: TObject);
    procedure B3Click(Sender: TObject);
    procedure B0Click(Sender: TObject);
    procedure BREMClick(Sender: TObject);
    procedure BRCLClick(Sender: TObject);
    procedure BBFOClick(Sender: TObject);
    procedure BFREQClick(Sender: TObject);
    procedure BISBClick(Sender: TObject);
    procedure BUSBClick(Sender: TObject);
    procedure CBXAGCChange(Sender: TObject);
    procedure CBXBWChange(Sender: TObject);
    procedure CBXMODEChange(Sender: TObject);
    procedure CBXMODEMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure CHANMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);
    procedure CHANMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormKeyPress(Sender: TObject; var Key: char);
    procedure FormResize(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure MBSCANWClick(Sender: TObject);
    procedure MPORTPROLClick(Sender: TObject);
    procedure MTCPClick(Sender: TObject);
    procedure RA1792ServerDisconnect(aSocket: TLSocket);
    procedure TNETTimer(Sender: TObject);
    procedure RA1792ServerAccept(aSocket: TLSocket);
    procedure RA1792ServerConnect(aSocket: TLSocket);
    procedure RA1792ServerError(const amsg: string; aSocket: TLSocket);
    procedure RA1792ServerReceive(aSocket: TLSocket);
    procedure MDEFSTClick(Sender: TObject);
    procedure M57600BClick(Sender: TObject);
    procedure M9600BClick(Sender: TObject);
    procedure M38400BClick(Sender: TObject);
    procedure MABOUTClick(Sender: TObject);
    procedure MENALLClick(Sender: TObject);
    procedure MKHZTClick(Sender: TObject);
    procedure MMAGALLFClick(Sender: TObject);
    procedure MMANCHClick(Sender: TObject);
    procedure MMANSTClick(Sender: TObject);
    procedure MRSTATEClick(Sender: TObject);
    procedure MRXADDClick(Sender: TObject);
    procedure MSGDblClick(Sender: TObject);
    procedure MBlueClick(Sender: TObject);
    procedure MBlackClick(Sender: TObject);
    procedure MGNULICClick(Sender: TObject);
    procedure MLSTATEClick(Sender: TObject);
    procedure MMANClick(Sender: TObject);
    procedure MExitClick(Sender: TObject);
    procedure MSHOWCClick(Sender: TObject);
    procedure MSSTATEClick(Sender: TObject);
    procedure MSmeterClick(Sender: TObject);
    procedure MGreenClick(Sender: TObject);
    procedure MSTBITEClick(Sender: TObject);
    procedure MSTBITEFMClick(Sender: TObject);
    procedure RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
            Shift: TShiftState; X, Y: Integer);
    procedure RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure SHKNOB1MouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure SHKNOBMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);
    procedure SPFREQChange(Sender: TObject);
    procedure SPFREQMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure TBATTENChange(Sender: TObject);
    procedure TSMDISPTimer(Sender: TObject);
    procedure TSmeterTimer(Sender: TObject);
    procedure B1Click(Sender: TObject);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState;
            WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure M115200BClick(Sender: TObject);
    procedure M19200BClick(Sender: TObject);
    procedure MCustomClick(Sender: TObject);
    procedure MttyUSB0Click(Sender: TObject);
    procedure MttyUSB1Click(Sender: TObject);
    procedure KeyboardHandler(keyp: string);
    procedure DispRXf;
    procedure UpdateControls;
    function SendCommand(cmd, par: string): boolean;
    function OpenRemote: boolean;
    function CloseRemote: boolean;
    function SetRXFreq: boolean;
    procedure GetStatus;
    procedure LoadChannel(Sender: TObject);
    procedure RestoreState;
    procedure SaveConfig;
    procedure ReadConfig;
    procedure Message(ms: string);
    procedure DisableAllControls;
    procedure EnableAllControls;
    procedure SetActiveButtons;
    procedure CloseProgram;
    procedure LoadChannelsDir;
    procedure UpdateDisplay;
    procedure ParseStatus(Status: string);
    procedure DisableAllButtons;
    procedure DisableKeypad;
    procedure EnableKeypad;
    procedure EnableCmdKeys;
    procedure EnableCombosAndEdits;
    procedure SpinEditUpdateOnly(Sender: TObject; value: integer);
    function AdjustDecimalSeparator(s: string): string;
    function GetReply: boolean;
    procedure ReadChanFile;
    procedure SaveChanFile;
    procedure SetColors(AColor: TCOlor);
    procedure GetLevel;
    procedure SetKeyCaptions(Mode: integer);
    function GetRxstring: string;
    procedure PutRXString(s: string);
    function GetChanString: string;
    procedure PutChanString(s: string);
    procedure MFONTMAGNClick(Sender: TObject);
    procedure FontMagnify;
    function ShowControlChars(cmd: string): string;
    procedure UPDDISPTimer(Sender: TObject);

  private
    { private declarations }
    procedure DoMessages;
  public
    { public declarations }
  end;

Const
  //
  // RA1792 command values (listener)
  //
  Set_Frequency:   string = 'F';     // in MHz downto 1 Hz, e.g. F12.345678
  Set_Mode:        string = 'D';     // 1..6 (AM, FM, CW, ISB, USB, LSB)
  Set_Bandwidth:   string = 'I';     // 1..5 (0.3, 1, 3, 6, 16 kHz)
  Set_AGC:         string = 'M';     // 1..7 (short, medium, long, manual,
                                     // short variable, medium variable,
                                     // long variable
  Set_BFO:         string = 'B';     // in kHz, 10 Hz step, range -8...+8 kHz
                                     // e.g. B1.23
  Set_Attenuation: string = 'A';     // IF attenuation, 3 digits, 000..150
  Start_BITE:      string = 'X';     // Starting BITE number, 2 digits

  //
  // RA1792 command values (talker)
  //
  Get_Status:       string = 'G';    // Get receiver status (all parameters)
  Get_Parameters:   string = 'T';    // Select returned parameters
                                     // (F, D, I, M, B, A, R)

  // PROLOGIX GPIP-USB Interface commands
  P_GetSetAddr:     string = '++addr';    // Set GPIB address
  P_GetSetMode:     string = '++mode';    // Set GPIB mode
  P_ResetInterface: string = '++rst';     // Reset GPIB interface

  // Filenames
  ConfigFileName: string = 'Config.dat';
  RXChanFileName: string = 'RXChan.dat';
  AUXFilename:    string = 'AUXconf.dat';

  // Mode values
  M_AM  = 1;
  M_FM  = 2;
  M_CW  = 3;
  M_ISB = 4;
  M_LSB = 5;
  M_USB = 6;

  // Filter values
  FILTER_300    = 1;
  FILTER_1000   = 2;
  FILTER_3200   = 3;
  FILTER_6000   = 4;
  FILTER_16000  = 5;

  // AGC values
  AGC_SHORT      = 1;
  AGC_MEDIUM     = 2;
  AGC_LONG       = 3;
  AGC_MAN        = 4;
  AGC_MAN_SHORT  = 5;
  AGC_MAN_MEDIUM = 6;
  AGC_MAN_LONG   = 7;

  // frequency coverage
  MINRXF = 0;
  MAXRXF = 29999999;

  // special characters
  EOL: string = #$0d+#$0a;
  stLF: string = #$0a;
  CR: char = #$0d;
  LF: char = #$0a;
  MeterChars = '»»»»»»»»»»»»';

  // TRUE/FALSE Constants
  ENABLED:  boolean = TRUE;
  DISABLED: boolean = FALSE;
  NOERROR:  boolean = TRUE;
  ERROR:    boolean = FALSE;
  NUMKEYS:  boolean = TRUE;
  FUNKEYS:  boolean = FALSE;

  // Timeout & delay values
  READTIMEOUT = 1000 * OneMillisecond;
  Delay100ms = 100 * OneMillisecond;

  // Keyboard states
  STATE_POWEROFF = 0;
  STATE_NORMAL   = 1;
  STATE_SETFREQ  = 2;
  STATE_SETBFO   = 3;
  STATE_SELCHAN  = 4;
  STATE_STOCHAN  = 5;
  STATE_SCAN     = 6;
  STATE_BITE     = 7;

  // Keyboard labels modes
  KBMODE_FUNKEYS   = 1;
  KBMODE_NUMKEYS   = 2;
  KBMODE_SCANDELAY = 3;

  // Display background color
  DispBackColor: TColor = $0a74ff;

  // Object resize array max components number
  MAXCOMPONENTS = 250;

Type
  TRXState = record
      Freq_rx: longint;
      Freq_step: integer;
      Mode_RX: integer;
      Filter: integer;
      AGC: integer;
      BFO: integer;
      Scan: boolean;
      IFGain: integer;
      BITEErr: integer;
      State: integer;
      Tune: boolean;
      ScanFlag: boolean;
    end;

    TRXChannels = array[0..99] of TRXState;

var
  RA1792: TRA1792;
  ProgVersion: TVersionQuad;
  Major, Minor, Revision, Build: integer;
  ShortVersion, LongVersion: string;
  RXChannels: TRXChannels;
  RXState, OldRXState: TRXState;
  SerPort: TSerialHandle = 0;
  SerPortName: string = '/dev/ttyS0';
  SerPortSpeed: integer = 115200;
  SleepTime: integer;
  TimeOneCharMs: integer;
  RXAddress: integer = 1;
  s,t, dBm: integer;
  Status: string = '';
  sCHtmp: string;
  iCHnum: integer;
  LastChan: integer = 0;
  rCurChan: real;
  CurChan: integer;
  Enable_status: boolean = FALSE;
  In_Wheel: boolean = FALSE;
  In_Status: boolean = FALSE;
  In_RXFreqMouseUp: boolean = FALSE;
  In_TXFreqMouseUp: boolean = FALSE;
  In_ChanMouseWheel: boolean = FALSE;
  In_ChanMouseUp: boolean = FALSE;
  In_Command: boolean = FALSE;
  Update_Only: boolean = FALSE;
  tmp: longint;
  HomeDir,StateDir,BandsDir,ChannelsDir,BandScansDir,DocDir, GPLv3Dir: string;
  timeout: TTime;
  LEDColorOFF: TColor = clGreen;
  LEDColorON: TColor = ClLime;
  sLEDColor: string = 'green';
  HaveConfig: boolean = FALSE;
  n,m: integer;
  BandFileName: string;
  RemoteMode: string;
  CmdTag: integer = -1;
  PrevTag: integer = -1;
  DigitCoord: array[1..9, 1..2] of integer;
  sf1,sd,si,sm,sb,sa,sx,sr,ss: string;
  KeyboardState: integer;
  DigitWidth: integer;
  prevalpha: real = 0;
  r, cx, cy,X2, Y2, R2: real;
  KnobOffsetX, KnobOffsetY: integer;
  HzPerTurn: integer = 1000;
  SignalLevel: integer = 40;
  OKPressed: boolean;
  ScanDelay: TTime = 2*Delay100ms;
  LastTuningTime: TTime;
  PPI_X, PPI_Y: integer;
  FontMagn: integer = 100;
  ErrMsg: string;

  {$IFDEF AUTOSCALE}
  // Variables used by autoscale code
  OH, OW: integer;
  Xs: array[0..MAXCOMPONENTS] of integer;
  Ys: array[0..MAXCOMPONENTS] of integer;
  Ws: array[0..MAXCOMPONENTS] of integer;
  Hs: array[0..MAXCOMPONENTS] of integer;
  Wt: array[0..MAXCOMPONENTS] of Integer;
  Ht: Integer;
  Wc: array[0..MAXCOMPONENTS] of Integer;
  Hc: Integer;
  Wn: array[0..MAXCOMPONENTS] of Integer;
  Hn: Integer;
  Hf: array[0..MAXCOMPONENTS] of Integer;
  {$ENDIF}
  PrevHeight, PrevWidth: integer;
  StartHeight, StartWidth: integer;
  StartX, StartY: integer;

  // Variables for TNET rigctl server
  RXLine: string;
  NumTCPConnections: integer = 0;
  RemFreqChange: boolean = FALSE;
  PanelUpdateRequired: boolean = FALSE;
  MsgToShow: string  = '';


resourcestring
  // Messages
     CANTOPENCHAN = 'Can''t open channel file.';
     CANTOPENFILE = 'Can''t open file.';
     CANTREADCHAN = 'Can''t read channels file.';
    CANTWRITECHAN = 'Can''t write channels file.';
  CANTWRITECONFIG = 'Can''t write config file.';
   CANTWRITESTATE = 'Can''t write state file.';
          CHANMGR = 'Channel Manager';
        CHECKINST = 'Check your installation.';
         CLOSEPGM = 'Close program';
              CMD = 'Cmd: ';
    COMMANDFAILED = 'Command failed.';
      COPYRIGHTST = ' (C) 2012-2023 G. Perotti, I1EPJ';
      DEFAULTSTEP = 'Default step ';
     DOCFILESNOTF = 'Manual and/or license files not found.';
      ENTERCHANNO = 'Enter channel# (0..99).';
     ENTERFREQKHZ = 'Enter frequency in kHz.';
         ERRORNET = 'Net error occurred: %s';
         ERRORMSG = 'Error';
     GETPROGVFAIL = 'GetProgramVersion failed, please enable'+LineEnding+
                    'version informations in Project Options.';
    INTINITFAILED = 'Interface init failed.';
      LASTBITERES = 'Last BITE result: ';
     LISTENFAILED = 'TRPserver socket: listen() failed';
       MRXADDCAPT = 'Receiver address (%d)';
        NOAUXFILE = 'No AUX file saved.';
      NOCHANSCANF = 'No channels with SCAN flag set found.';
      NORXXRECORD = 'No Rxx record present in returned status, level reading disabled.';
        NOTRREPLY = 'No reply to TR command, level reading disabled.';
       NOVERSINFO = 'No version info.';
   OLDSTYLECONFIG = 'Old Style config file found, updating and saving.';
       REALLYEXIT = 'Really exit?';
       RXCANTTUNE = 'RX can''t tune to that frequency.';
     SCANDELAYMSG = 'Scan delay';
       SETBFOFREQ = 'Set BFO frequency';
         STATEMGR = 'State manager';
      WHENBITEFIN = 'When BITE finishes, press ENTER.';

implementation

{$R *.lfm}

{ TRA1792 }

//
// Auxiliary procedures & functions
//

// Make control chars visible in command strings
function TRA1792.ShowControlChars(cmd: string): string;

var chp: integer;
    stmp: string = '';
    chtmp: string;
begin
  // Display control chars in command strings (see MSHOWCClick)
  for chp := 1 to length(cmd) do begin
    case cmd[chp] of
      {$IFDEF WIN32}
      // No control chars unicode symbols in Win32 (XP)
      // Show only CR as paragraph symbol
      #$0D: chtmp := '¶';
      otherwise chtmp := cmd[chp];
      {$ELSE}
      #$01: chtmp := '␁';
      #$02: chtmp := '␂';
      #$03: chtmp := '␃';
      #$04: chtmp := '␄';
      #$06: chtmp := '␆';
      #$07: chtmp := '␇';
      #$0A: chtmp := '␊';
      #$0D: chtmp := '␍';
      #$10: chtmp := '␐';
      #$15: chtmp := '␕';
      #$18: chtmp := '␘';
      otherwise chtmp := cmd[chp];
      {$ENDIF}
    end;
    stmp := stmp+chtmp;
  end;
  ShowControlChars := stmp;
end;

function TRA1792.GetRXString: string;

begin
 GetRXString := Trim(RXFreq10M.Caption+RXFreq1M.Caption+RXFreq100k.Caption+
                RXFreq10k.Caption+RXFreq1k.Caption+RXFreqDOT.Caption+RXFreq100H.Caption+
                RXFreq10H.Caption+RXFreq1H.Caption);
end;

//Put RX string - Put a string into RX frequency display right-aligned
procedure TRA1792.PutRXString(s: string);

var st: string;

begin
 st := s;
 if Length(st) < 9 then st := RightStr('         '+s,9);
 RXFreq10M.Caption := st[1];
 RXFreq1M.Caption := st[2];
 RXFreq100k.Caption := st[3];
 RXFreq10k.Caption := st[4];
 RXFreq1k.Caption := st[5];
 RXFreqDOT.Caption := st[6];
 RXFreq100H.Caption := st[7];
 RXFreq10H.Caption := st[8];
 RXFreq1H.Caption := st[9];
end;

// Put a 2 char string into Channel number TLabels
procedure TRA1792.PutChanString(s:string);
begin
  CHANTENS.Caption := s[1];
  CHANUNITS.Caption := s[2];
end;

// Get a 2 char string from Channel number TLabels
function TRA1792.GetChanString:string;
begin
  GetChanString:= CHANTENS.Caption + CHANUNITS.Caption;
end;

// Adjust decimal separator in real number string
// Latest compilers do not accept anymore both separators (. and ,)
function TRA1792.AdjustDecimalSeparator(s: string): string;

var i: integer;

begin
  for i := 1 to Length(s) do
    if s[i]='.' then s[i] := DefaultFormatSettings.DecimalSeparator;
  AdjustDecimalSeparator := s;
end;

// Process message queue
procedure TRA1792.DoMessages;
begin
  Application.ProcessMessages;
end;

// Open communications
function TRA1792.OpenRemote: boolean;

var tmp: string;
    i,nc,len: integer;

begin
  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}

  // If selected, init Prologix USB-to-GPIB interface

  if MPORTPROL.Checked then begin
    tmp := P_GetSetMode+'1'+EOL;
    len := Length(tmp);
    {$IFDEF DEBUG_PROLOGIX}
    Message('PROLOGIX: '+ShowControlChars(tmp));
    {$ENDIF}
    nc := 0;
    for i := 1 to len do begin
      nc := nc + SerWrite(SerPort, tmp[i], 1);
      {$IFDEF LINUX}
      sleep(1);
      {$ENDIF}
      {$IFDEF WINDOWS}
      // under windows minimum sleep is 15ms
      sleep(0);
      {$ENDIF}
    end;
    if nc <> len then begin
      Message(INTINITFAILED);
      exit(ERROR);
    end;

    tmp := P_GetSetAddr+IntToStr(RXAddress)+EOL;
    len := Length(tmp);
    {$IFDEF DEBUG_PROLOGIX}
    Message('PROLOGIX: '+ShowControlChars(tmp));
    {$ENDIF}
    nc := 0;
    for i := 1 to len do begin
      nc := nc + SerWrite(SerPort, tmp[i], 1);
      {$IFDEF LINUX}
      sleep(1);
      {$ENDIF}
      {$IFDEF WINDOWS}
      // under windows minimum sleep is 15ms
      sleep(0);
      {$ENDIF}
    end;
    if nc <> len then begin
      Message(INTINITFAILED);
      exit(ERROR);
    end;
  end else begin
    // RA1792 RS232 interface init: what to do?
  end;

  // Check for RA1792 responding. Issue G command and wait for status reply
  Result := SendCommand(Get_Status,'');
  if Result then begin
    if GetReply then
      OpenRemote := NOERROR
    else
      OpenRemote := ERROR;
  end;
end;

// Close communications
function TRA1792.CloseRemote: boolean;
begin
    // ANYTHING TO DO?
    CloseRemote := TRUE;
end;

//Get reply to commands
function TRA1792.GetReply: boolean;

var buf: char;
    timeout: TTime;
    nc: integer;

begin
  buf := chr(0);
  Status := '';
  repeat
    timeout := time + READTIMEOUT;
    repeat
      nc := SerRead(SerPort, buf, 1);
      {$IFDEF LINUX}
      sleep(1);
      {$ENDIF}
      {$IFDEF WINDOWS}
      // under windows minimum sleep is 15ms
      sleep(0);
      {$ENDIF}
    until (nc > 0) or (time > timeout);
    Status := Status + buf;
  until (buf = LF) or (time > timeout);

  if time > timeout then GetReply := ERROR else GetReply := NOERROR;
end;

// Send a command to RA1792
function TRA1792.SendCommand(cmd, par: string): boolean;

var tmp: string;
    i,nc,len: integer;
    ok: boolean;

begin
  Enable_Status := DISABLED;

  tmp := cmd+par+EOL;
  len := Length(tmp);

  if MSHOWC.Checked then Message(ShowControlChars(tmp));
    DoMessages;

  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}

  ok := NOERROR;
  nc := 0;
  for i := 1 to len do begin
    nc := nc + SerWrite(SerPort, tmp[i], 1);
    {$IFDEF LINUX}
    sleep(1);
    {$ENDIF}

    {$IFDEF WINDOWS}
    sleep(0);
    {$ENDIF}
  end;
  if nc <> len then begin
    Ok := ERROR;
    {$IFDEF TEST_SENDCOMMAND}
    Message('SerWrite returned ' + IntToStr(nc));
    {$ENDIF}
  end;

  if MSmeter.Checked then Enable_status := ENABLED;
  SendCommand := ok;
end;

// Get status
procedure TRA1792.GetStatus;

begin
  {$IFDEF TEST_USER_INTERFACE}
  status := 'F 18.070000D 3I 2M 7B 0.60A 000X 00S 1';
  exit;
  {$ENDIF}
  if SendCommand(Get_Status, '') then begin
    GetReply;
    ParseStatus(Status);
    {$IFDEF TEST_GETSTATUS}
    Message(Status);
    {$ENDIF}
    UpdateDisplay;
  end;
end;

// Get signal level
procedure TRA1792.GetLevel;

var i: integer;

begin
  {$IFDEF TEST_GETLEVEL}
    SignalLevel := Random(120);
    exit;
  {$ENDIF}
  if SendCommand(Get_Parameters, 'R') then begin
    GetReply;
    {$IFDEF TEST_USER_INTERFACE}
    Status := 'R 100'; // don't know if it is really so...
    {$ENDIF}
    // My RA1792 does not have the TR command, so this may likely not work...
    i := pos('R', Status);
    if i > 0 then begin
      sr := copy(Status, i+2, 3);
      SignalLevel := StrToInt(sr);
    end else begin
      sr := '';
      SignalLevel := 0;
      if Status = '' then
        Message(NOTRREPLY);
      if (Status <> '') and (i=0) then begin
        Message(NORXXRECORD);
        {$IFDEF DEBUG_SMETER}
        Message('Reply was: '+Status);
        {$ENDIF}
      end;
      TSmeter.Enabled := FALSE;
      TSMDISP.Enabled := FALSE;
      MSmeter.Checked := FALSE;
      Enable_Status := FALSE;
    end;
  end;
end;

// Parse status
procedure TRA1792.ParseStatus(Status: string);

var i: integer;
begin
  // Status is something like 'F 18.070000D 3I 2M 7B 0.60A 000X 00S 1'
  //                           00000000011111111112222222222333333333
  //                           12345678901234567890123456789012345678
  // Don't know the R status format (no TR command on my RA1792)
  i := pos('F', Status);
  if i > 0 then begin
    sf1 := copy(Status, i+2, 9);
    RXState.Freq_rx := Trunc(StrToFloat(AdjustDecimalSeparator(sf1))*1000000);
//    PutRXString(sf);
  end;

  i := pos('D', Status);
  if i > 0 then begin
    sd := copy(Status, i+2, 1);
    RXState.Mode_RX := StrToInt(sd);
  end else sd := '';

  i := pos('I', Status);
  if i > 0 then begin
    si := copy(Status, i+2, 1);
    RXState.Filter := StrToInt(si);
  end else si := '';

  i := pos('M', Status);
  if i > 0 then begin
    sm := copy(Status, i+2, 1);
    RXState.AGC := StrToInt(sm);
  end else sm := '';

  i := pos('B', Status);
  if i > 0 then begin
    sb := copy(Status, i+2, 4);
    RXState.BFO := Trunc(StrToFloat(AdjustDecimalSeparator(sb))*1000);
  end else sb := '';

  i := pos('A', Status);
  if i > 0 then begin
    sa := copy(Status, i+2, 3);
    RXState.IFGain := StrToInt(sa);
    TBAtten.Position := RXstate.IFGain;
  end else sa := '';

  i := pos('X', Status);
  if i > 0 then begin
    sx := copy(Status, i+2, 2);
    RXState.BITEErr := StrToInt(sx);
  end else sx := '';

  i := pos('S', Status);
  if i > 0 then begin
    ss := copy(Status, i+2, 1);
    RXState.State := StrToInt(ss);
  end else ss := '';

  // My RA1792 does not have the TR command, so this may likely not work...
  i := pos('R', Status);
  if i > 0 then begin
    sr := copy(Status, i+2, 3);
    SignalLevel := StrToInt(sr);
  end else begin
    sr := '';
    SignalLevel := 0;
  end;
end;

// UPDDISP display
procedure TRA1792.UpdateDisplay;

var s: string;

begin
  DispRXF;

  LBFM.Visible := FALSE;
  LBCW.Visible := FALSE;
  LBAM.Visible := FALSE;
  LBUSB.Visible := FALSE;
  LBLSB.Visible := FALSE;
  LBISB.Visible := FALSE;
  LBBFOFR.Visible := FALSE;
  LBBFO1.Visible := FALSE;
  LBBFO2.Visible := FALSE;
  LBSHORT.Visible := FALSE;
  LBMED.Visible := FALSE;
  LBLONG.Visible := FALSE;

  case RXState.Mode_RX of
    M_FM:  LBFM.Visible := TRUE;
    M_AM:  LBAM.Visible := TRUE;
    M_CW:  begin
      LBCW.Visible := TRUE;
      LBBFOFR.Visible := TRUE;
      LBBFO1.Visible := TRUE;
      LBBFO2.Visible := TRUE;
    end;
    M_LSB: LBLSB.Visible := TRUE;
    M_USB: LBUSB.Visible := TRUE;
    M_ISB: LBISB.Visible := TRUE;
  end;

  case RXState.AGC of
    AGC_SHORT: begin
      LBSHORT.Visible := TRUE;
      LBMED.Visible := FALSE;
      LBLONG.Visible := FALSE;
      LBMAN.Visible := FALSE;
      TBATTEN.Enabled := FALSE;
    end;
    AGC_MEDIUM: begin
      LBSHORT.Visible := FALSE;
      LBMED.Visible := TRUE;
      LBLONG.Visible := FALSE;
      LBMAN.Visible := FALSE;
      TBATTEN.Enabled := FALSE;
    end;
    AGC_LONG: begin
      LBSHORT.Visible := FALSE;
      LBMED.Visible := FALSE;
      LBLONG.Visible := TRUE;
      LBMAN.Visible := FALSE;
      TBATTEN.Enabled := FALSE;
    end;
    AGC_MAN_SHORT: begin
      LBSHORT.Visible := TRUE;
      LBMED.Visible := FALSE;
      LBLONG.Visible := FALSE;
      LBMAN.Visible := TRUE;
      TBATTEN.Enabled := TRUE;
    end;
    AGC_MAN_MEDIUM: begin
      LBSHORT.Visible := FALSE;
      LBMED.Visible := TRUE;
      LBLONG.Visible := FALSE;
      LBMAN.Visible := TRUE;
      TBATTEN.Enabled := TRUE;
    end;
    AGC_MAN_LONG: begin
      LBSHORT.Visible := FALSE;
      LBMED.Visible := FALSE;
      LBLONG.Visible := TRUE;
      LBMAN.Visible := TRUE;
      TBATTEN.Enabled := TRUE;
    end;
  end;

  if RXstate.Tune then
    LBTUNE.Visible := TRUE
  else
    LBTUNE.Visible := FALSE;

  if si <> '' then begin
    LBBW.Visible := TRUE;
    LBBW1.Visible := TRUE;
    LBBW2.Visible := TRUE;
  end else begin
    LBBW.Visible := FALSE;
    LBBW1.Visible := FALSE;
    LBBW2.Visible := FALSE;
  end;

  case RXstate.Filter of
    FILTER_300:  LBBW.Caption := '0.3';
    FILTER_1000: LBBW.Caption := '1.0';
    FILTER_3200: LBBW.Caption := '3.2';
    FILTER_6000: LBBW.Caption := '6.0';
    FILTER_16000: LBBW.Caption := '16';
  end;

  Str(RXState.BFO/1000:4:2, s);
  if RXState.BFO >= 0 then s := '+'+s;
    LBBFOFR.Caption := s;

  case KeyboardState of
    STATE_NORMAL: begin
      STIND.Visible := FALSE;
      LBSCAN.Visible := FALSE;
    end;
    STATE_SELCHAN: begin
      STIND.Visible := TRUE;
      LBSCAN.Visible := FALSE;
      if RXState.ScanFlag then
        LBSCAN.Visible := TRUE
      else
        LBSCAN.Visible := FALSE;
      end;
    STATE_BITE: begin
      Message(LASTBITERES+IntToStr(RXstate.BITEErr));
      KeyboardState := STATE_NORMAL;
      RXState := OldRXState;
      RestoreState;
    end;
  end;
  UpdateControls;
end;

// Set RX frequency
function TRA1792.SetRXFreq: boolean;

var  sf: string;
     rf: real;

begin
  rf := RXState.Freq_rx / 1000000;
  str(rf:9:6,sf);
  SendCommand(Set_Frequency, sf);
  OldRXState := RXState;
  SetRXFreq := NOERROR;
end;

// Display a message in MSG memo
procedure TRA1792.Message(ms: string);

var p: integer;
    tmp: string;

begin
  tmp := '';
  for p := 1 to length(ms) do begin
    if ms[p] = chr(13) then
      tmp := tmp+'[CR]'
    else if ms[p] = chr(10) then
      tmp := tmp+'[LF]'
    else
      tmp := tmp+ms[p];
  end;
  MSG.Lines.Add(tmp);
  DoMessages;
end;

// Clear messages in MSG memo
procedure TRA1792.MSGDblClick(Sender: TObject);
begin
  MSG.Lines.Clear;
end;

// Display only RX frequency
procedure TRA1792.DispRXf;

var tmp: real;
    s: string;

begin
  tmp := RXState.Freq_rx / 1000;
  str(tmp:9:3,s);
  PutRXString(s);
  SpinEditUpdateOnly(SPFREQ, RXState.Freq_RX);
end;

// UPDDISP non-button controls
procedure TRA1792.UpdateControls;

begin
  with RXState do begin
    SpinEditUpdateOnly(SPFREQ, Freq_rx);
    CBXMODE.ItemIndex := Mode_RX - 1;
    if CBXBW.Enabled then
      CBXBW.ItemIndex := Filter - 1
    else
      CBXBW.ItemIndex := -1;
    CBXAGC.ItemIndex := AGC - 1;
  end;
end;

// UPDDISP SpinEdit display only
procedure TRA1792.SpinEditUpdateOnly(Sender: TObject; value: integer);

var ASpin: TSpinEditEx;

begin
  ASpin := Sender as TSpinEditEx;
  Update_Only := TRUE;
  ASpin.Value := Value;
  Update_Only := FALSE;
end;

// Keyboard handler
procedure TRA1792.KeyboardHandler(keyp: string);

var p: integer;
    s: string;
    f: longint;

begin
  case KeyboardState of
    STATE_NORMAL: {Nothing to do};
    STATE_SETFREQ: begin
      s := GetRXString;
      p := Pos('-',s);
      if p > 0 then begin
        s[p] := keyp[1];
        if s[p+1] = '.' then p := p+2 else p := p+1;
        f := trunc(1000*StrToFloat(AdjustDecimalSeparator(s)));
        if f > MAXRXF then begin
          Message(RXCANTTUNE);
          exit;
        end;
        if p <= Length(s) then s[p] := '-';
          PutRXString(s);
      end;
    end;
    STATE_SELCHAN, STATE_STOCHAN: begin
      sCHtmp[iCHNum] := keyp[1];
      iCHNum := iCHNum + 1;
      if iCHnum > 2 then iCHNum := 1;
      PutChanString(sCHtmp);
    end;
  end;
end;

// Save RX channels file to disk
procedure TRA1792.SaveChanFile;

var RXChanFile: file of TRXChannels;

begin
  AssignFile(RXChanFile, HomeDir+RXChanFileName);
  try
    Rewrite(RXChanFile);
    Write(RXChanFile, RXChannels);
    CloseFile(RXChanFile);
  except
    Message(CANTWRITECHAN);
  end;
end;

// Read RX channels file from disk
procedure TRA1792.ReadChanFile;

var RXChanFile: file of TRXChannels;

begin
  AssignFile(RXChanFile, HomeDir+RXChanFileName);
  try
    Reset(RXChanFile);
    Read(RXChanFile, RXChannels);
    CloseFile(RXChanFile);
  except
    Message(CANTREADCHAN);
  end;
end;

// Save config to disk
procedure TRA1792.SaveConfig;

var ConfigFile: Text;
    w, h, l, t: integer;

begin
  AssignFile(ConfigFile, HomeDir+ConfigFileName);
  try
    Rewrite(ConfigFile);
    WriteLn(ConfigFile, SerPortName);
    Writeln(ConfigFile, SerPortSpeed);
    Writeln(ConfigFile, RXAddress);

    WriteLn(ConfigFile, MSmeter.Checked);
    WriteLn(ConfigFile,MSHOWC.Checked);
    WriteLn(ConfigFile,MENALL.Checked);

    WriteLn(ConfigFile,sLEDColor);

    if M1Hz.Checked then WriteLn(ConfigFile, '1');
    if M10Hz.Checked then WriteLn(ConfigFile, '10');
    if M100Hz.Checked then WriteLn(ConfigFile, '100');
    if M1kHz.Checked then WriteLn(ConfigFile, '1000');

    if MTIMED.Checked then
      WriteLn(ConfigFIle, '0')
    else
      WriteLn(ConfigFile, IntToStr(HzPerTurn));

      w := Width;
      h := Height;
      l := Left;
      t := Top;

      // Under XP and maybe some other system Top and Left may go negative,
      // so stay on the safe side and check them all.
      if w > Screen.Width then w := Screen.Width;
      if h > Screen.Height then h := Screen.Height;
      if l < 0 then l := 0;
      if t < 0 then t := 0;

      WriteLn(ConfigFile, w);
      WriteLn(ConfigFile, h);
      Writeln(ConfigFile, l);
      Writeln(ConfigFile, t);

      {$IFDEF DEBUG_POSANDSIZE}
      Message('W='+IntToStr(w));
      Message('H='+IntToStr(h));
      Message('L='+IntToStr(l));
      Message('T='+IntToStr(t));
      {$ENDIF}

      WriteLn(ConfigFile,FontMagn);

      WriteLn(ConfigFile, MPORTPROL.Checked);

      WriteLn(ConfigFile,RA1792Server.Host+':'+IntTostr(RA1792Server.Port));

      CloseFile(ConfigFile);
    except
        Message(CANTWRITECONFIG);
    end;
end;

// Read config from disk
procedure TRA1792.ReadConfig;

var ConfigFile: Text;
    s: string;
    i,p: integer;

begin
  AssignFile(ConfigFile, HomeDir+ConfigFileName);
  try
    Reset(ConfigFile);
    ReadLn(ConfigFile, SerPortName);
    {$IFDEF WINDOWS}
    // The \\.\ syntax is optional for ports 1..9 but is needed
    // for ports > 9, so insert it anyway if not present
    if pos('\\.\', SerPortName) = 0 then
       SerPortName := '\\.\' + SerPortName;
    {$ENDIF}
    ReadLn(ConfigFile, SerPortSpeed);
    ReadLn(ConfigFile, RXAddress);
    ReadLn(ConfigFile, s);
    MSMeter.Checked := StrToBool(s);
    ReadLn(ConfigFile, s);
    MSHOWC.Checked := StrToBool(s);
    ReadLn(ConfigFile,s);
    MENALL.Checked := StrToBool(s);
    ReadLn(ConfigFile,sLEDColor);
    ReadLn(ConfigFile, s);
    RXState.Freq_step := StrToInt(s);

    if sLEDColor = 'black' then begin
      LEDColorON := clBlack;
      MBlack.Checked := TRUE;
    end;
    if sLEDColor = 'green' then begin
      LEDColorON := clGreen;
      MGreen.Checked := TRUE;
    end;
    if sLEDColor = 'blue' then begin
      LEDColorON := clBlue;
      MBlue.Checked := TRUE;
    end;

    for i := 1 to ComponentCount - 1 do begin
      if Components[i] is TLabel then
        TLabel(Components[i]).font.color := LEDCOlorON;
    end;

    case RXState.Freq_step of
         1: M1Hz.Checked := TRUE;
        10: M10Hz.Checked := TRUE;
       100: M100Hz.Checked := TRUE;
      1000: M1kHz.Checked := TRUE;
      otherwise begin
        M1Hz.Checked := TRUE;
        RXState.Freq_Step := 1;
      end;
    end;

    ReadLn(ConfigFile, s);
    HzPerTurn := STrToInt(s);
    case HzPerTurn of
      0:      MTIMED.Checked := TRUE;
      500:    M0P5KHZT.Checked := TRUE;
      1000:   M1KHZT.Checked := TRUE;
      10000:  M10KHZT.Checked := TRUE;
      100000: M100KHZT.Checked := TRUE;
      otherwise begin
        HzPerTurn := 1000;
        M1KHZT.Checked := TRUE;
      end;
    end;

    //Read starting dimension and position
    ReadLn(ConfigFile, StartWidth);
    ReadLn(ConfigFile, StartHeight);
    ReadLn(ConfigFile, StartX);
    ReadLn(ConfigFile, StartY);

    //Read font magnification factor
    ReadLn(configFile,s);
    if not TryStrToInt(s,FontMagn) then begin
      ShowMessage(OLDSTYLECONFIG);
      CloseFile(ConfigFile);
      FontMagn := 100;
      SaveConfig;
    end;

    // Read Prologix option
    ReadLn(configFile,s);
    MPORTPROL.Checked := s='TRUE';
    if MPORTPROL.Checked then
      MRXADD.Visible := TRUE
    else
      MRXADD.Visible := FALSE;

    // Read TCP server address:port
    ReadLn(ConfigFile,s);
    if s <> '' then begin
      p := Pos(':',s);
      if p>0 then begin
        RA1792Server.Host := Copy(s,1,p-1);
        RA1792Server.Port := StrToInt(Copy(s,p+1));
      end;
    end;

    CloseFile(ConfigFile);

    MMAG100.Checked := FALSE;
    MMAG110.Checked := FALSE;
    MMAG125.Checked := FALSE;
    MMAG150.Checked := FALSE;
    MMAG175.Checked := FALSE;
    MMAG200.Checked := FALSE;
    case FontMagn of
      100: MMAG100.Checked := TRUE;
      110: MMAG110.Checked := TRUE;
      125: MMAG125.Checked := TRUE;
      150: MMAG150.Checked := TRUE;
      175: MMAG175.Checked := TRUE;
      200: MMAG200.Checked := TRUE;
      otherwise MMAG100.Checked := TRUE;
    end;


    RXFreq1H.Font.Color := LEDColorON;
    RXFreq1H.Font.Color := LEDColorON;
    RXFreq10H.Font.Color := LEDColorON;
    RXFreq100H.Font.Color := LEDColorON;
    RXFreq1k.Font.Color := LEDColorON;
    RXFreq10k.Font.Color := LEDColorON;
    RXFreq100k.Font.Color := LEDColorON;
    RXFreq1M.Font.Color := LEDColorON;
    RXFreq10M.Font.Color := LEDColorON;
    RXFreqDOT.Font.Color := LEDColorON;
    CHANTENS.Font.Color := LEDColorON;
    CHANUNITS.Font.Color := LEDColorON;
    STREMOTE.Font.Color := DispBackColor;
    LBBFO.Font.Color := DispBackColor;
    LBTUNE.Font.Color := DispBackColor;
    LBSCAN.Font.Color := DispBackColor;
    LBNETW.Font.Color := DispBackColor;
    LBNETW.Color := LEDColorON;
    STREMOTE.Color := LEDColorON;
    LBBFO.Color := LEDColorON;
    LBTUNE.Color := LEDColorON;
    LBSCAN.Color := LEDColorON;
    LBFM.Font.Color := DispBackColor;
    LBAM.Font.Color := DispBackColor;
    LBCW.Font.Color := DispBackColor;
    LBUSB.Font.Color := DispBackColor;
    LBLSB.Font.Color := DispBackColor;
    LBISB.Font.Color := DispBackColor;
    LBFM.Color := LEDColorON;
    LBAM.Color := LEDColorON;
    LBCW.Color := LEDColorON;
    LBUSB.Color := LEDColorON;
    LBLSB.Color := LEDColorON;
    LBISB.Color := LEDColorON;
    LBMET.Font.Color := LEDColorON;
    LBSHORT.Font.Color := DispBackColor;
    LBMED.Font.Color := DispBackColor;
    LBLONG.Font.Color := DispBackColor;
    LBMAN.Font.Color := DispBackColor;
    LBSHORT.Color := LEDColorON;
    LBMED.Color := LEDColorON;
    LBLONG.Color := LEDColorON;
    LBMAN.Color := LEDColorON;
  except
    // default values
    {$IFDEF LINUX}
    if pos(SerPortName,'/dev/') = 0 then
      // No valid port
      SerPortName := '/dev/ttyUSB0';
    {$ENDIF}

    {$IFDEF WINDOWS}
    if pos(SerPortName,'\\.\COM') = 0 then
      SerPortName := '\\.\COM1';
    {$ENDIF}
    SerPortSpeed := 115200;
    MSmeter.Checked := FALSE;
    LEDColorOFF := $ff740a;
    LEDColorON := clBlack;
    sLEDColor := 'black';
    MGreen.Checked := FALSE;
    MBlack.Checked := TRUE;
    MBlue.Checked := FALSE;
    M115200B.Checked := TRUE;
    M1KHZ.Checked := TRUE;
    M1KHZT.Checked := TRUE;
    MMAG100.Checked := TRUE;
    MPORTPROL.Checked := TRUE;

    // Default window placement if no config file
    StartWidth :=  1000;
    StartHeight := 335;
    StartX := (Screen.Width-StartWidth) div 2;
    StartY := (Screen.Height - StartHeight) div 2;
  end;
  {$IFDEF LINUX}
  if pos(SerPortName, 'ttyUSB0') > 0 then begin
  {$ENDIF}

  {$IFDEF WINDOWS}
  if pos(SerPortName, 'COM1') > 0 then begin
  {$ENDIF}
    MttyUSB0.Checked := TRUE;
    MttyUSB1.Checked := FALSE;
    MCustom.Checked := FALSE;
  end else begin
    {$IFDEF LINUX}
    if pos(SerPortName, 'ttyUSB1') > 0 then begin
    {$ENDIF}
    {$IFDEF WINDOWS}
    if pos(SerPortName, 'COM2') > 0 then begin
    {$ENDIF}
      MttyUSB0.Checked := FALSE;
      MttyUSB1.Checked := TRUE;
      MCustom.Checked := FALSE;
    end else begin
      MttyUSB0.Checked := FALSE;
      MttyUSB1.Checked := FALSE;
      MCustom.Checked := TRUE;
    end;
  end;

  if SerPortSpeed = 9600 then M9600B.Checked := TRUE;
  if SerPortSpeed = 19200 then M19200B.Checked := TRUE;
  if SerPortSpeed = 38400 then M38400B.Checked := TRUE;
  if SerPortSpeed = 57600 then M57600B.Checked := TRUE;
  if SerPortSpeed = 115200 then M115200B.Checked := TRUE;
  MRXADD.Caption := Format(MRXADDCAPT,[RXAddress]);
  HaveConfig := TRUE;
end;

// Restore state in RXState
procedure TRA1792.RestoreState;

var par: string;
    r: real;

begin
  // Set RX frequency
  SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);
  r := RXState.Freq_rx / 1000000;
  str(r:9:6,par);
  SendCommand(Set_Frequency, par);

  // Set mode
  if RXState.Mode_RX in [M_AM..M_USB] then begin
    par := IntToStr(RXState.Mode_RX);
  end else begin
    // Invalid value, default to A1
    RXState.Mode_RX := M_CW;
    par := '1';
  end;
  CBXMode.ItemIndex := RXState.Mode_RX-1;
  if not SendCommand(Set_Mode, par) then
    Message(COMMANDFAILED);
  if (RXState.Mode_RX in [M_LSB, M_USB, M_ISB]) then
    CBXBW.Enabled := FALSE
  else
    CBXBW.Enabled := TRUE;

  // Set filter
  if CBXBW.Enabled then begin
    if RXState.Filter in [Filter_300..Filter_16000] then begin
      CBXBW.ItemIndex := RXState.Filter-1;
      par := IntToStr(RXState.Filter);
    end else begin
      // Invalid filter, assume 1000 Hz
      RXstate.Filter := Filter_1000;
      par := '2';
    end;

    if not SendCommand(Set_Bandwidth, par) then
      Message(COMMANDFAILED);
    end;

    // Set AGC
    if RXState.AGC in[AGC_SHORT..AGC_MAN_LONG] then begin
      par := IntToStr(RXstate.AGC);
      CBXAGC.ItemIndex := RXState.AGC-1;
    end else begin
      // Invalid value, default to FAST
      RXState.AGC := AGC_LONG;
      par := '1';
    end;
    if not SendCommand(SET_AGC, par) then
      Message(COMMANDFAILED);

    // Status polling
    if MSmeter.Checked then begin
      TSmeter.Enabled := TRUE;
      In_Command := FALSE;
    end else begin
      TSmeter.Enabled := FALSE;
    end;

    // Scan mode
    if RXState.Scan then begin
      //TODO?
    end;

    // TUNE
    if RXstate.Tune then begin
      LBTUNE.Visible := TRUE;
      SPFREQ.Enabled := TRUE;
    end else begin
      LBTUNE.Visible := FALSE;
      SPFREQ.Enabled := FALSE;
    end;

    // Frequency step
    case RXstate.Freq_step of
        1: M1HZ.Checked := TRUE;
       10: M10HZ.Checked := TRUE;
      100: M100HZ.Checked := TRUE;
     1000: M1KHZ.Checked := TRUE;
    end;

    // Get and display new parameters
    GetStatus;
    UpdateDisplay;
    SetActiveButtons;
end;

//Create ordered Channels submenus
procedure TRA1792.LoadChannelsDir;

var m: TMenuItem;
    n: integer;
    Info: TSearchRec;
    FileList: TStringList;

begin
  FileList := TStringList.Create;
  FileList.Sorted := TRUE;
  if FindFirst(ChannelsDir+'*.dat', faAnyFile, info) = 0 then begin
    repeat
      delete(Info.Name, Length(Info.Name)-3, 4);
      FileList.Add(Info.Name);
    until FindNext(Info) <> 0;
    FindClose(Info);
    MChannels.Clear;
    for n := 0 to FileList.Count-1 do begin
      m := TmenuItem.Create(MChannels);
      m.Caption := FileList.Strings[n];
      m.OnClick := @LoadChannel;
      MChannels.Add(m);
    end;
    FileList.Clear;
    FileList.Free;
  end else begin
    MChannels.Clear;
    m := TmenuItem.Create(MChannels);
    m.Caption := 'No channels defined';
    MChannels.Add(m);
  end;
end;

// Load a channel file
procedure TRA1792.LoadChannel(Sender: TObject);

var BandFile: FILE of TRXState;

begin
  In_Command := TRUE;
  AssignFile(BandFile,ChannelsDir+(Sender as TMenuItem).Caption+'.dat');
  try
    Reset(BandFile);
    Read(BandFile,RXState);
    CloseFile(BandFile);
  except
    Message(CANTOPENCHAN);
  end;
  RestoreState;
  UpdateDisplay;
  In_Command := FALSE;
end;

// Disable all controls except REM and some Menus
procedure TRA1792.DisableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if (Components[i] is TLabel) then
      TLabel(Components[i]).Visible := FALSE;
    if (Components[i] is TControl) then
      TControl(Components[i]).Enabled := FALSE;
  end;

  // Disable Menus requiring connection with the receiver...
  MLState.Enabled := FALSE;
  MSState.Enabled := FALSE;
  MSCHANN.Enabled := FALSE;
  MRSTATE.Enabled := FALSE;
  MSmeter.Enabled := FALSE;
  MChannels.Enabled := FALSE;

  //...and enable those who don't
  MPORT.Enabled := TRUE;
  MSPEED.Enabled := TRUE;
  MRXADD.Enabled := TRUE;
  MttyUSB0.Enabled := TRUE;
  MttyUSB1.Enabled := TRUE;
  MCustom.Enabled := TRUE;
  M9600B.Enabled := TRUE;
  M19200B.Enabled := TRUE;
  M38400B.Enabled := TRUE;
  M57600B.Enabled := TRUE;
  M115200B.Enabled := TRUE;
  BREM.Enabled := TRUE;
  RXFreq1H.Enabled := TRUE;
  RXFreq10H.Enabled := TRUE;
  RXFreq100H.Enabled := TRUE;
  RXFreqDOT.Enabled := TRUE;
  RXFreq1k.Enabled := TRUE;
  RXFreq10k.Enabled := TRUE;
  RXFreq100k.Enabled := TRUE;
  RXFreq1M.Enabled := TRUE;
  RXFreq10M.Enabled := TRUE;
  LBCHAN.Enabled := TRUE;
end;

// Disable all buttons
procedure TRA1792.DisableAllButtons;

var i: integer;

begin
  if MENALL.Checked then exit;
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TJButton then
      TJButton(Components[i]).Enabled := FALSE;
    if Components[i] is TSpinEditEx then
      TSpinEditEx(Components[i]).Enabled := FALSE;
    if Components[i] is TComboBoxEx then
      TComboBoxEx(Components[i]).Enabled := FALSE;
  end;
end;

// Disable numeric keypad buttons
procedure TRA1792.DisableKeypad;

begin
  if MENALL.Checked then exit;
  B0.Enabled := FALSE;
  B1.Enabled := FALSE;
  B2.Enabled := FALSE;
  B3.Enabled := FALSE;
  B4.Enabled := FALSE;
  B5.Enabled := FALSE;
  B6.Enabled := FALSE;
  B7.Enabled := FALSE;
  B8.Enabled := FALSE;
  B9.Enabled := FALSE;
  BSTORE.Enabled := FALSE;
  BENTER.Enabled := FALSE;
end;

// Enable numeric keypad buttons
procedure TRA1792.EnableKeypad;

begin
  B0.Enabled := TRUE;
  B1.Enabled := TRUE;
  B2.Enabled := TRUE;
  B3.Enabled := TRUE;
  B4.Enabled := TRUE;
  B5.Enabled := TRUE;
  B6.Enabled := TRUE;
  B7.Enabled := TRUE;
  B8.Enabled := TRUE;
  B9.Enabled := TRUE;
end;

// Enable command keys
procedure TRA1792.EnableCmdKeys;

begin
  BREM.Enabled := TRUE;
  BRCL.Enabled := TRUE;
  BBFO.Enabled := TRUE;
  BUSB.Enabled := TRUE;
  BCHAN.Enabled := TRUE;
  BFREQ.Enabled := TRUE;
  BISB.Enabled := TRUE;
  BLSB.Enabled := TRUE;
  BTUNE.Enabled := TRUE;
  BAM.Enabled := TRUE;
  BCW.Enabled := TRUE;
  BFM.Enabled := TRUE;
  BCHSCAN.Enabled := TRUE;
end;

// Enable Combo and Edit boxes
procedure TRA1792.EnableCombosAndEdits;
begin
  SPFREQ.Enabled := TRUE;
  CBXMODE.Enabled := TRUE;
  CBXBW.Enabled := TRUE;
  CBXAGC.Enabled := TRUE;
end;

// Set numbers or functions key captions
procedure TRA1792.SetKeyCaptions(Mode: integer);

begin
  case Mode of
    KBMODE_FUNKEYS: begin
      // Function keys captions
      B0.LCaption.Caption := 'AUX';
      B1.LCaption.Caption := 'BW1';
      B2.LCaption.Caption := 'BW2';
      B3.LCaption.Caption := 'BW3';
      B4.LCaption.Caption := 'BW4';
      B5.LCaption.Caption := 'BW5';
      B6.LCaption.Caption := 'MAN';
      B7.LCaption.Caption := 'SHORT';
      B8.LCaption.Caption := 'MED';
      B9.LCaption.Caption := 'LONG';
    end;
    KBMODE_NUMKEYS: begin
      // Numeric keys captions
      B0.LCaption.Caption := '0';
      B1.LCaption.Caption := '1';
      B2.LCaption.Caption := '2';
      B3.LCaption.Caption := '3';
      B4.LCaption.Caption := '4';
      B5.LCaption.Caption := '5';
      B6.LCaption.Caption := '6';
      B7.LCaption.Caption := '7';
      B8.LCaption.Caption := '8';
      B9.LCaption.Caption := '9';
    end;
    KBMODE_SCANDELAY: begin
      B0.LCaption.Caption := '0.1s';
      B1.LCaption.Caption := '0.2s';
      B2.LCaption.Caption := '0.5s';
      B3.LCaption.Caption := '0.8s';
      B4.LCaption.Caption := '1s';
      B5.LCaption.Caption := '2s';
      B6.LCaption.Caption := '4s';
      B7.LCaption.Caption := '6s';
      B8.LCaption.Caption := '8s';
      B9.LCaption.Caption := '10s';
    end;
  end;
end;

// Enable all controls
procedure TRA1792.EnableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
  	TControl(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TMenuItem then begin
        TMenuItem(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TSpinEditEx then begin
        TSpinEditEx(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TComboBoxEx then begin
        TComboBoxEx(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TShape then begin
        TShape(Components[i]).Visible := TRUE;
    end;
  end;
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
  end;
  RXFreq1H.Visible := TRUE;
  RXFreq10H.Visible := TRUE;
  RXFreq100H.Visible := TRUE;
  RXFreqDOT.Visible := TRUE;
  RXFreq1k.Visible := TRUE;
  RXFreq10k.Visible := TRUE;
  RXFreq100k.Visible := TRUE;
  RXFreq1M.Visible := TRUE;
  RXFreq10M.Visible := TRUE;
  STREMOTE.Visible := TRUE;
  LBTUNE.Visible := RXState.Tune;
  LBSCALE.Visible := TRUE;
  LBRF.Visible := TRUE;
  LBDBU.Visible := TRUE;
  LBRF.Visible := TRUE;
  LBMET.Visible := TRUE;
  LBKHZ.Visible := TRUE;
  LBFREQ.Visible := TRUE;
  DoMessages;
end;

// Set active/not active buttons & edit state
procedure TRA1792.SetActiveButtons;

begin
  if MENALL.Checked then
      // Do not try to disable inactive buttons, so do nothing and return
      exit
  else begin
    case KeyboardState of
      STATE_NORMAL: begin
        EnableAllControls;
        EnableKeypad;
        if not RXState.Tune then SPFREQ.Enabled := FALSE;
        if not (RXState.Mode_RX in [M_AM, M_CW, M_FM]) then begin
          CBXBW.Enabled := FALSE;
          CBXBW.ItemIndex := -1; // none
        end else begin
          CBXBW.Enabled := TRUE;
          CBXBW.ItemIndex := RXState.Filter + 1;
        end;
        if RXState.Mode_RX in [M_USB, M_LSB] then begin
          B4.Enabled := FALSE;
          B5.Enabled := FALSE;
        end;
      end;
      STATE_SETFREQ: begin
        DisableAllButtons;
        EnableKeypad;
        BRCL.Enabled := TRUE;
        BFREQ.Enabled := TRUE;
        BENTER.Enabled := TRUE;
      end;
      STATE_SELCHAN: begin
        DisableAllButtons;
        EnableKeypad;
        BRCL.Enabled := TRUE;
        BCHSCAN.Enabled := TRUE;
        BENTER.Enabled := TRUE;
      end;
      STATE_STOCHAN: begin
        DisableAllButtons;
        EnableKeypad;
        BENTER.Enabled := TRUE;
      end;
      STATE_SETBFO: begin
        DisableAllButtons;
        BBFO.Enabled := TRUE;
        BENTER.Enabled := TRUE;
        BREM.Enabled := TRUE;
      end;
      STATE_SCAN: begin
        BREM.Enabled := FALSE;
        BBFO.Enabled := FALSE;
        BTUNE.Enabled := FALSE;
        BSTORE.Enabled := FALSE;
        BENTER.Enabled := FALSE;
        BFREQ.Enabled := FALSE;
        BCHAN.Enabled := FALSE;
        BRCL.Enabled := FALSE;
        BISB.Enabled := FALSE;
        BLSB.Enabled := FALSE;
        BUSB.Enabled := FALSE;
        BAM.Enabled := FALSE;
        BCW.Enabled := FALSE;
        BFM.Enabled := FALSE;
        BCHSCAN.Enabled := TRUE;
      end;
      STATE_BITE: begin
        DisableAllButtons;
        BENTER.Enabled := TRUE;
        Message(WHENBITEFIN);
      end;
    end;
  end;
  UpdateControls;
end;


//
// Object init & functions
//

// Various initializations at startup
procedure TRA1792.FormCreate(Sender: TObject);

var i: integer;

begin
  {$IFDEF TEST_USER_INTERFACE}
  RA1792.Caption := 'RA1792 Control - USER INTERFACE TEST MODE!';
  {$ELSE}
  RA1792.Caption := 'RA1792 Control ' + ShortVersion;
  {$ENDIF}

  {$IFDEF WINDOWS}
  // Serial ports names
  MttyUSB0.Caption := 'COM1';
  MttyUSB1.Caption := 'COM2';
  {$ENDIF}

  // Check X and Y PPI
  PPI_X := Graphics.ScreenInfo.PixelsPerInchX;
  PPI_Y := Graphics.ScreenInfo.PixelsPerInchY;
  if PPI_X <> PPI_Y then
    ShowMessage('This screen does not have square pixels:'+
    LineEnding+'PPI_X='+IntTostr(PPI_X)+' PPI_Y='+IntTostr(PPI_Y)+
    LineEnding+'Expect strange behaviour in scaling routines.');

  PutRXString('');
  LBMET.Caption := '';
  DisableAllControls;
  DisableAllButtons;
  BREM.Enabled := TRUE;
  MSmeter.Enabled := FALSE;
  MOTHER.Enabled := FALSE;
  KeyboardState := STATE_POWEROFF;

  // Read channels directory
  LoadChannelsDir;

  // Read config
  ReadConfig;

  // Calculate fixed parameters for tuning knob
  r := (SHKNOB1.Width - SHKNOB.Width) / 2;
  cx := SHKNOB1.Left+(SHKNOB1.Width - SHKNOB.Width) / 2;
  cy := SHKNOB1.Top+(SHKNOB1.Height -SHKNOB.Height) / 2;
  X2 := SHKNOB1.Width / 2;
  Y2 := SHKNOB1.Height / 2;
  R2 := SHKNOB1.Width / 2;

  SPFREQ.Enabled := FALSE;

  // Component & window scaling
  {$IFDEF AUTOSCALE}
  for i := 0 to ComponentCount - 1 do
    if Components[i] is TControl then begin
      Xs[i] := TControl(Components[i]).Left;
      Ys[i] := TControl(Components[i]).Top;
      Ws[i] := TControl(Components[i]).Width;
      Hs[i] := TControl(Components[i]).Height;
    if (Components[i] is TLabel) or
       (Components[i] is TMemo) or
       (Components[i] is TSpinEditEx) or
       (Components[i] is TComboBoxEx) then begin
      Hf[i] := TControl(Components[i]).Font.Height;
    end;
    if (Components[i] is TJButton) then begin
      Hf[i] := (Components[i] as TJButton).LCaption.Font.Height;
    end;
  end;
  {$IFDEF DEBUG_AUTOSCALE}
  Message('total of '+IntToStr(i)+' components');
  {$ENDIF}

  // Figure out available height since now ClientHeight=Height
  {$IFDEF LINUX}
  OH := CBXAGC.Top+CBXAGC.Height+1;
  OW := ClientWidth;
  {$ENDIF}
  {$IFDEF WINDOWS}
  OH := CBXAGC.Top+CBXAGC.Height+5;
  OW := ClientWidth+5;
  {$ENDIF}

  {$IFDEF DEBUG_AUTOSCALE}
  Message('H='+IntTostr(Height)+' W='+IntTostr(Width));
  Message('OH='+IntToStr(OH)+' OW='+IntToSTR(OW));
  {$ENDIF}
  {$ENDIF}
end;

// Close down program
procedure TRA1792.CloseProgram;
begin
  if MSmeter.Checked then begin
    TSmeter.Enabled := FALSE;
    Enable_Status := FALSE;
  end;
  if SerPort > 0 then begin
    SerDrain(SerPort);
    SerClose(SerPort);
  end;
  if HaveConfig then SaveConfig;
  Halt;
end;

// Closing by clicking on X
procedure TRA1792.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  CloseAction := caNone;
  MExitClick(Sender);
end;

// Allows to enter commands using the keyboard
procedure TRA1792.FormKeyPress(Sender: TObject; var Key: char);

var f: integer;

begin
  if In_Status or SPFREQ.Focused then exit;

  {$IFDEF TEST_KEYPRESS}
  Message('Key pressed: '+''''+Key+'''');
  {$ENDIF}

  case LowerCase(Key) of
    'u': BUSBCLick(Sender);
    'l': BLSBClick(Sender);
    'c': BCWClick(Sender);
    'a': BAMClick(Sender);
    'f': BFMClick(Sender);
    'i': BISBClick(Sender);
    'b': begin
      RXstate.Filter := RXstate.Filter+1;
      if RXstate.Filter > FILTER_16000 then RXState.Filter := FILTER_300;
      RestoreState;
      UpdateDisplay;
    end;
    '0': B0Click(Sender);
    '1': B1Click(Sender);
    '2': B2Click(Sender);
    '3': B3Click(Sender);
    '4': B4Click(Sender);
    '5': B5Click(Sender);
    '6': B6Click(Sender);
    '7': B7Click(Sender);
    '8': B8Click(Sender);
    '9': B9Click(Sender);
    '*': BFREQCLick(Sender);
    #$0D: BENTERCLick(Sender);
    '+': begin
      if RXstate.Tune then begin
        f := RXState.Freq_rx + RXState.Freq_step;
        if f <= MAXRXF then begin
          RXState.Freq_rx := f;
          SetRXFreq;
          UpdateDisplay;
        end;
      end;
    end;
    '-': begin
      if RXstate.Tune then begin
        f := RXstate.Freq_rx - RXstate.Freq_step;
        if f >= MINRXF then begin
          RXState.Freq_rx := f;
          SetRXFreq;
          UpdateDisplay;
        end;
      end;
    end;
    '/': begin
      if M1HZ.Checked then MDEFSTClick(M10HZ)
      else if M10HZ.Checked then MDEFSTClick(M100HZ)
      else if M100HZ.Checked then MDEFSTClick(M1KHZ)
      else if M1KHZ.Checked then MDEFSTClick(M1HZ);
    end;
    '.': BTUNEClick(Sender);
  end;
  key := #0;
end;

// Main form resize
procedure TRA1792.FormResize(Sender: TObject);

var i: integer;
    c: TControl;
    NewHeight, NewHeight1, NewHeight2: integer;
    alpha, myx, myy: real;

begin
  {$IFDEF AUTOSCALE}
  { Here's actual component scaling. }
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
      c := TControl(Components[i]);
      c.Left   := Xs[i] * ClientWidth div OW;
      c.Top    := Ys[i] * ClientHeight div OH;
      c.Width  := Ws[i] * ClientWidth div OW;
      c.Height := Hs[i] * ClientHeight div OH;
      if c is TComboBoxEx then
        (c as TComboBoxEx).ItemHeight := Hs[i] * ClientHeight div OH;
      {$IFDEF DEBUG_AUTOSCALE}
      Message('h='+IntTostr(c.Height)+' was '+IntTostr(Hs[i]));
      {$ENDIF}
    end;
    c := TControl(Components[i]);
    if (c is TLabel) or
       (c is TJButton) or
       (c is TMemo) or
       (c is TSpinEditEx) or
       (c is TComboBoxEx) then begin
      {$IFDEF DEBUG_AUTOSCALE}
      Message('scaling font');
      {$ENDIF}
      // Scale using ClientWidth since main screen is
      // more wide than high
      NewHeight1 := Hf[i] * ClientWidth div OW;
      NewHeight2 := Hf[i] * ClientHeight div OH;
      NewHeight := min(abs(NewHeight1), abs(NewHeight2));

      if c is TJButton then
        (c as TJButton).LCaption.Font.Height := NewHeight
      else
        c.Font.Height := NewHeight;
    end;
    {$IFDEF DEBUG_AUTOSCALE}
    Message('newheight1='+inttostr(NewHeight1)+' newheight2='+inttostr(NewHeight2));
    Message('newheight='+inttostr(NewHeight));
    Message('Width='+IntToSTr(Width)+', Height='+IntToStr(Height));
    Message('ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight));
    {$ENDIF}
  end;

  // Recalculate parameters for tuning knob
  r := (SHKNOB1.Width - SHKNOB.Width) / 2.3;
  cx := SHKNOB1.Left+(SHKNOB1.Width - SHKNOB.Width) / 2;
  cy := SHKNOB1.Top+(SHKNOB1.Height -SHKNOB.Height) / 2;
  X2 := SHKNOB1.Width / 2;
  Y2 := SHKNOB1.Height / 2;
  R2 := SHKNOB1.Width / 2;

  // Put knob in the new position
  alpha := arctan2(SHKNOB1.Top-SHKNOB1.Height / 2, SHKNOB1.Left + SHKNOB1.Width / 2);
  myx := r * cos(alpha) + cx;
  myy := r * sin(alpha) + cy;
  SHKNOB.Top := trunc(myy);
  SHKNOB.Left := trunc(myx);
  {$IFDEF WINDOWS}
  CBXMODE.ClearSelection;
  CBXBW.ClearSelection;
  CBXAGC.ClearSelection;
  UpdateDisplay;
  {$ENDIF}
  {$ENDIF}
end;

// Main form things to do when shown
procedure TRA1792.FormShow(Sender: TObject);

begin
  // Restore saved width & height
  RA1792.Height := StartHeight;
  RA1792.Width := StartWidth;

  // Adapt main window to the screen height/width
  if RA1792.Height > Screen.Height then
    RA1792.Height := Screen.Height;
  if RA1792.Width > Screen.Width then
    RA1792.Width := Screen.Width;

  //Put window in the saved position
  if StartX <= Screen.Width - RA1792.Width then
    RA1792.Left := StartX
  else
    RA1792.Left := Screen.Width - RA1792.Width;

  if StartY <= Screen.Height - RA1792.Height then
    RA1792.Top := StartY
  else
    RA1792.Top := Screen.Height - RA1792.Height;

  {$IFDEF TEST_USER_INTERFACE}
  GetStatus;
  UpdateDisplay;
  EnableCmdKeys;
  EnableCombosAndEdits;
  SPFREQ.Enabled := FALSE;
  {$ENDIF}
end;

procedure TRA1792.MBSCANWClick(Sender: TObject);
begin
  FSCAN.Show;
end;

// if port is a Prologix one, enable specific code and menus
procedure TRA1792.MPORTPROLClick(Sender: TObject);
begin
  if MPORTPROL.Checked then
    MRXADD.Visible := TRUE
  else
    MRXADD.Visible := FALSE;
end;


// hamlib NET rigctl-more-or-less-compatible server networking code (LNET)
// Remote commands handling code is in UParse,pas, UCapabilities.pas and UState.pas
// The server code is so far incomplete, it's a little more than the bare minimum
// needed to work with WSJTX, FLDIGI, GRIG, XDX and XLOG.
// See also the comments in UParse.pas

// Accept a connection
procedure TRA1792.RA1792ServerAccept(aSocket: TLSocket);

begin
  NumTCPConnections := NumTCPConnections+1;
end;

procedure TRA1792.RA1792ServerConnect(aSocket: TLSocket);
begin

end;

// Handle disconnect received from client
procedure TRA1792.RA1792ServerDisconnect(aSocket: TLSocket);
begin
  if NumTCPConnections > 0 then NumTCPConnections := NumTCPConnections - 1;
end;

// Error handling
procedure TRA1792.RA1792ServerError(const amsg: string; aSocket: TLSocket);
begin
  Message(Format(ERRORNET,[amsg]));
  UpdateDisplay;
end;

// Receive data from client
procedure TRA1792.RA1792ServerReceive(aSocket: TLSocket);

var len: integer;

begin
  len := aSocket.GetMessage(RXLine);
  {$ifdef DEBUG_NET}
  Message('Received '+IntToStr(len)+' bytes');
  {$endif}
  if len > 0 then begin
    {$ifdef DEBUG_NET}
    Message('Net RX:'+ShowControlChars(RXLine));
    {$endif}
    ParseNetMsg(RXLine,aSocket);
  end;
end;

// Timer to show network messages and status from uparse.pas
procedure TRA1792.TNETTimer(Sender: TObject);
begin
  if NumTCPConnections > 0 then
    LBNETW.Visible := TRUE
  else
    LBNETW.Visible := FALSE;

  if MsgToShow <> '' then begin
    Message(MsgToShow);
    MsgToShow := '';
  end;
end;

//
// Menu handling
//

// "File" menu handling

// "Load state"
procedure TRA1792.MLSTATEClick(Sender: TObject);

var StateFile: FILE of TRXState;

begin
  OpenDialog1.InitialDir := StateDir;
  if OpenDialog1.Execute then begin
    AssignFile(StateFile,OpenDialog1.Filename);
    try
      Reset(StateFile);
      Read(StateFile,RXState);
      CloseFile(StateFile);
    except
      Message(CANTOPENFILE);
    end;
    RestoreState;
    UpdateControls;
  end;
end;

// "Save state" or "Save channel"
procedure TRA1792.MSSTATEClick(Sender: TObject);

var StateFile: file of TRXState;
    SModeRX: string = '';
    sMtmp,sFtmp: string;

begin
  case RXState.Mode_RX of
    M_ISB: SModeRX := 'ISB';
    M_USB: SModeRX := 'USB';
    M_LSB: SmodeRX := 'LSB';
    M_AM:  SModeRX := 'AM';
    M_CW:  SModeRX := 'CW';
    M_FM:  SModeRX := 'FM';
  end;

  sMtmp := sModeRX;
  sFtmp := GetRXString;

  if Sender = MSSTATE then SaveDialog1.InitialDir := StateDir;
  if Sender = MSCHANN then SaveDialog1.InitialDir := ChannelsDir;
  SaveDialog1.FileName := sFtmp + '-' + sMtmp + '.dat';
  if SaveDialog1.Execute then begin
    AssignFile(StateFile,SaveDialog1.Filename);
    try
      Rewrite(StateFile);
      Write(StateFile,RXState);
      CloseFile(StateFile);
    except
      Message(CANTWRITESTATE);
    end;
    if Sender = MSCHANN then LoadChannelsDir;
  end;
end;

// "Manage channels"
procedure TRA1792.MMANCHClick(Sender: TObject);
begin
    UMGR.Caption := CHANMGR;
    UMGR.TFLB.Directory := ChannelsDir;
    UMGR.TFLB.UpdateFileList;
    UMGR.Tag := 1;
    UMGR.SPMOVE2CH.Visible := FALSE;
    UMGR.SPMOVE2ST.Visible := TRUE;
    UMGR.Show;
end;

// "Manage States"
procedure TRA1792.MMANSTClick(Sender: TObject);
begin
    UMGR.Caption := STATEMGR;
    UMGR.TFLB.Directory := StateDir;
    UMGR.TFLB.UpdateFileList;
    UMGR.Tag := 0;
    UMGR.SPMOVE2CH.Visible := TRUE;
    UMGR.SPMOVE2ST.Visible := FALSE;
    UMGR.Show;
end;

// "Restore current state"
procedure TRA1792.MRSTATEClick(Sender: TObject);

begin
    RestoreState;
    EnableAllControls;
    UpdateControls;
    UpdateDisplay;
    SetActiveButtons;
end;

// "Exit"
procedure TRA1792.MExitClick(Sender: TObject);
begin
    SaveConfig;
    SaveChanFile;
    if MessageDlg(CLOSEPGM,REALLYEXIT,mtConfirmation,
                  [mbYES, mbNO],0) = mrYES then CloseProgram;
end;

//
// "Other functions" menu handling
//

// "Start BITE"
procedure TRA1792.MSTBITEClick(Sender: TObject);
begin
    OldRXState := RXState;
    SendCommand(Start_BITE,'001');
    KeyboardState := STATE_BITE;
    SetActiveButtons;
end;

// "Start BITE from..."
procedure TRA1792.MSTBITEFMClick(Sender: TObject);

var StartTestNo, err: integer;

begin
    OldRXState := RXState;
    repeat
        OKPressed := FALSE;
        FEntTestNo.Show;
        Val(FEntTestno.EDTESTNO.Text, StartTestNo, err);
        DoMessages;
    until OKPressed and (err=0);
    SendCommand(Start_BITE, IntToStr(StartTestNo));
    KeyboardState := STATE_BITE;
    SetActiveButtons;
end;

//
// "Options" menu handling
//

// "RS232 port"

// "ttyUSB0"
procedure TRA1792.MttyUSB0Click(Sender: TObject);

begin
    {$IFDEF LINUX}
    SerPortName := '/dev/ttyUSB0';
    {$ENDIF}

    {$IFDEF WINDOWS}
    SerPortName := 'COM1';
    {$ENDIF}
    SaveConfig;
end;

// "ttyUSB1"
procedure TRA1792.MttyUSB1Click(Sender: TObject);

begin
    {$IFDEF LINUX}
    SerPortName := '/dev/ttyUSB1';
    {$ENDIF}

    {$IFDEF WINDOWS}
    SerPortName := 'COM2';
    {$ENDIF}
    SaveConfig;
end;

// "Custom..."
procedure TRA1792.MCustomClick(Sender: TObject);
begin
    FSerport.Show;
    FSerport.Eser.Text := SerPortName;
end;

// "RS232 speed"
// Does not matter for the Prologix USB-to-GPIB interface
// but may be required for other interfaces

// 9600 baud
procedure TRA1792.M9600BClick(Sender: TObject);
begin
    SerPortSpeed := 9600;
    SaveConfig;
    TimeOneCharMs := 11000 div SerPortSpeed;
    SleepTime := 70-(TimeOneCharMs);
end;

// "19200 baud"
procedure TRA1792.M19200BClick(Sender: TObject);
begin
    SerPortSpeed := 19200;
    SaveConfig;
    TimeOneCharMs := 11000 div SerPortSpeed;
    SleepTime := 70-(TimeOneCharMs);
end;

// "38400 baud"
procedure TRA1792.M38400BClick(Sender: TObject);
begin
    SerPortSpeed := 38400;
    SaveConfig;
    TimeOneCharMs := 11000 div SerPortSpeed;
    SleepTime := 70-(TimeOneCharMs);
end;

// "57600 baud"
procedure TRA1792.M57600BClick(Sender: TObject);
begin
    SerPortSpeed := 57600;
    SaveConfig;
    TimeOneCharMs := 11000 div SerPortSpeed;
    SleepTime := 70-(TimeOneCharMs);
end;

// "115200 baud"
procedure TRA1792.M115200BClick(Sender: TObject);
begin
    SerPortSpeed := 115200;
    SaveConfig;
    TimeOneCharMs := 11000 div SerPortSpeed;
    SleepTime := 70-(TimeOneCharMs);
end;

// "Receiver address"
procedure TRA1792.MRXADDClick(Sender: TObject);
begin
    FADDR.SPRXADD.Value := RXAddress;
    FADDR.Show;
end;

// TCP server address:port
procedure TRA1792.MTCPClick(Sender: TObject);
begin
  FTCP.EDADDPORT.Text := RA1792SERVER.Host+':'+IntToStr(RA1792SERVER.Port);
  FTCP.Show;
end;

// Set kHz per turn
procedure TRA1792.MKHZTClick(Sender: TObject);
begin
     HzPerTurn := (Sender as TmenuItem).Tag;
end;

procedure TRA1792.MMAGALLFClick(Sender: TObject);
begin
  FontMagnify;
end;

// Set default step (for mouse wheel and + - keys)
procedure TRA1792.MDEFSTClick(Sender: TObject);
begin
    RXstate.Freq_step := (Sender as TMenuItem).Tag;
    case RXState.Freq_step of
        1: M1HZ.Checked := TRUE;
        10: M10Hz.Checked := TRUE;
        100: M100HZ.Checked := TRUE;
        1000: M1KHZ.Checked := TRUE;
    end;
    Message(DEFAULTSTEP+IntTostr(RXState.Freq_step)+'Hz');
end;

// "Enable S-Meter"
procedure TRA1792.MSmeterClick(Sender: TObject);
begin
    if MSmeter.Checked then begin
        TSmeter.Enabled := TRUE;
        TSMDISP.Enabled := TRUE;
        Enable_Status := TRUE;
    end else begin
        TSmeter.Enabled := FALSE;
        TSMDISP.Enabled := FALSE;
        Enable_Status := FALSE;
        SignalLevel := 0;
        LBMET.Caption :='';
    end;
    SaveConfig;
    UpdateControls;

end;

// "LED Color"

// Set display colors
procedure TRA1792.SetColors(AColor: TColor);
begin
    RXFreq1H.Font.Color := AColor;
    RXFreq10H.Font.Color := AColor;
    RXFreq100H.Font.Color := AColor;
    RXFreqDOT.Font.COlor := AColor;
    RXFreq1k.Font.Color := AColor;
    RXFreq10k.Font.Color := AColor;
    RXFreq100k.Font.Color := AColor;
    RXFreq1M.Font.Color := AColor;
    RXFreq10M.Font.Color := AColor;
    CHANTENS.Font.Color := AColor;
    CHANUNITS.Font.Color := AColor;
    LBCHAN.Font.Color := AColor;
    LBFREQ.Font.Color := AColor;
    STREMOTE.Color := AColor;
    LBNETW.Color := AColor;
    LBKHZ.Font.Color := AColor;
    STIND.Font.Color := AColor;
    LBTUNE.Color := AColor;
    LBSCAN.Color := AColor;
    LBFM.Color := AColor;
    LBAM.Color := AColor;
    LBCW.Color := AColor;
    LBUSB.Color := AColor;
    LBLSB.Color := AColor;
    LBISB.Color := AColor;
    LBSHORT.Color := AColor;
    LBMAN.Color := AColor;
    LBMED.Color := AColor;
    LBLONG.Color := AColor;
    LBRF.Font.Color := AColor;
    LBMET.Font.Color := AColor;
    LBDBU.Font.Color := AColor;
    LBSCALE.Font.Color := AColor;
    LBBFO.Color := AColor;
    LBBFO1.Font.Color := AColor;
    LBBFO2.Font.Color := AColor;
    LBBFOFR.Font.color := AColor;
    LBBW.Font.Color := AColor;
    LBBW1.Font.Color := AColor;
    LBBW2.Font.Color := AColor;
    LBFAULT.Color := AColor;
    LBMUTE.COlor := AColor;
end;

// "Blue"
procedure TRA1792.MBlueClick(Sender: TObject);
begin
    MBlack.Checked := FALSE;
    MBlue.Checked := TRUE;
    MGreen.Checked := FALSE;
    LEDCOlorOFF := clGreen;
    LEDColorON := clBlue;
    sLEDColor := 'blue';
    SetColors(LEDColorON);
    SaveConfig;
    UpdateControls;
end;

// "Green"
procedure TRA1792.MGreenClick(Sender: TObject);
begin
    MBlack.Checked := FALSE;
    MBlue.Checked := FALSE;
    MGreen.Checked := TRUE;
    LEDCOlorOFF := clGray;
    LEDColorON := clGreen;
    sLEDColor := 'green';
    SetColors(LEDColorON);
    SaveConfig;
    UpdateControls;
end;

// "Black"
procedure TRA1792.MBlackClick(Sender: TObject);
begin
    MBlack.Checked := TRUE;
    MBlue.Checked := FALSE;
    MGreen.Checked := FALSE;
    LEDCOlorOFF := clGray;
    LEDColorON := clBlack;
    sLEDColor := 'black';
    SetColors(LEDColorON);
    SaveConfig;
    UpdateControls;
end;

// "Enable all controls"
procedure TRA1792.MENALLClick(Sender: TObject);
begin
    SaveConfig;
    if SerPort <> 0 then begin
        if MENALL.Checked then
            EnableAllControls
        else begin
            SetActiveButtons;
            UpdateControls;
        end;
    end;
end;

procedure TRA1792.MFONTMAGNClick(Sender: TObject);
begin
    FontMagn := (Sender as TMenuItem).tag;
    SaveConfig;
    FontMagnify;
end;

procedure TRA1792.FontMagnify;

var c: TControl;
    i, NewHeight1, NewHeight2, NewHeight: integer;

begin
    {$IFNDEF DISABLE_FONTSCALE}
    for i := 0 to ComponentCount-1 do begin
      if (Components[i] is TLabel) or
         (Components[i] is TStaticText) or
         (Components[i] is TMemo) or
         (Components[i] is TJButton) then begin
        {$IFDEF DEBUG_AUTOSCALE}
        Message('scaling font');
        {$ENDIF}
        c := TControl(Components[i]);
        if (c.Tag = 0) or MMAGALLF.Checked then begin
          NewHeight1 := Hf[i] * ClientWidth * FontMagn div (OW * 100);
          NewHeight2 := Hf[i] * ClientHeight * FontMagn div (OH * 100);
          NewHeight := min(abs(NewHeight1), abs(NewHeight2));
        end else begin
          NewHeight1 := Hf[i] * ClientWidth div OW;
          NewHeight2 := Hf[i] * ClientHeight div OH;
          NewHeight := min(abs(NewHeight1), abs(NewHeight2));
        end;
        if (Components[i] is TJButton) then begin
          {$IFDEF DEBUG_FONTSCALE}
          Message('OW='+inttostr(OW)+' OH='+inttostr(OH));
          Message('ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight));
          Message('Width='+IntToSTr(Width)+', Height='+IntToStr(Height));
          Message('NewHeight='+IntToStr(NewHeight));
          {$ENDIF}
          (Components[i] as TJButton).LCaption.Font.Height := NewHeight;
        end else begin
          c.Font.Height := NewHeight;
        end
      end;
    end;
    {$ENDIF}
end;

// "Show commands"
procedure TRA1792.MSHOWCClick(Sender: TObject);
begin
  SaveConfig;
end;


//
// "Channels" menu is dinamically built
// See procedure LoadChannelsDir
//

//
// "Help" menu handling
//

// "About"
procedure TRA1792.MABOUTClick(Sender: TObject);

begin
    FAbout.Show;
end;

// "Manual"
procedure TRA1792.MMANClick(Sender: TObject);

begin
    FMAN.Show;
end;

// "GNU License"
procedure TRA1792.MGNULICClick(Sender: TObject);
begin
    FGNU.Show;
end;

//
// Buttons handling (block 1)
//

// REM
procedure TRA1792.BREMClick(Sender: TObject);

var i: integer;

begin
    if In_Command then exit;
    In_Command := TRUE;
    if SerPort > 0 then begin
        if MSmeter.Checked then begin
          TSmeter.Enabled := FALSE;
          Enable_Status := FALSE;
          TSMDISP.Enabled := FALSE;
        end;
        CloseRemote;
        SerDrain(SerPort);
        SerClose(SerPort);
        SerPort := 0;
        SaveConfig;
        SaveChanFile;
        PutRXString('');
        TSmeter.Enabled := FALSE;
        SignalLevel := 0;
        LBMET.Caption := '';
        DisableAllControls;
        MSmeter.Enabled := FALSE;
        for i := 0 to ComponentCount - 1 do begin
            if Components[i] is TLabel then
                TLabel(Components[i]).Visible:= FALSE;
        end;
        CBXMODE.ItemIndex := -1;
        CBXBW.ItemIndex := -1;
        CBXAGC.ItemIndex := -1;
        MBSCANW.Enabled := FALSE;
        RA1792Server.Disconnect(TRUE);
        NumTCPConnections := 0;
        KeyboardState := STATE_POWEROFF;
    end else begin
        KeyboardState := STATE_NORMAL;
        MSG.Lines.Clear;
        ReadConfig;
        ReadChanFile;
        {$IFDEF TEST_USER_INTERFACE}
        SerPort := 1;
        {$ELSE}
	SerPort := SerOpen(SerPortName);
        {$ENDIF}
        if SerPort > 0 then begin
            SerSetParams(SerPort,SerPortSpeed,8,NoneParity,1,[]);
            SerSetDTR(SerPort, TRUE);
            SerSetRTS(SerPort, TRUE);

            if OpenRemote then begin
                {$IFDEF TEST_REMOTE}
                Message(Status);
                {$ENDIF}
                EnableAllControls;
                EnableKeypad;
                SetKeyCaptions(KBMODE_FUNKEYS);
                ParseStatus(Status);
                OldRXState := RXState;
                DoMessages;
                UpdateDisplay;
                SetActiveButtons;
                if MSmeter.Checked then begin
                    TSmeter.Enabled := TRUE;
                    Enable_Status := TRUE;
                    TSMDISP.Enabled := TRUE;
                end else begin
                    TSmeter.Enabled := FALSE;
                    Enable_Status := FALSE;
                    TSMDISP.Enabled := FALSE;
                    LBMET.Caption := '';
                    SignalLevel := 0;
                end;
                RXstate.Tune := FALSE;
                LBTUNE.Visible := FALSE;
                LBNETW.Visible := FALSE;
                SPFREQ.Enabled := FALSE;
                if (RXState.Mode_RX in [M_LSB, M_USB, M_ISB]) then
                    CBXBW.Enabled := FALSE;
                MBSCANW.Enabled := TRUE;
                FontMagnify;
                // Start rigctl server
                if not RA1792Server.Listen(RA1792Server.Port,RA1792Server.Host) then
                  Message(LISTENFAILED);
             end else begin
                SerDrain(SerPort);
                SerClose(SerPort);
                SerPort := 0;
                if MessageDlg('Error',
                    'Failed to initialize communications.'
                    +LineEnding
                    +'Retry?',
                    mtConfirmation, [mbYES, mbNO],0) = mrNO then begin
                    CloseProgram;
               end else begin
                 In_Command := FALSE;
                 BREMClick(Sender);
               end;
            end;
	    end else begin
            if MessageDlg('Error',
                'Unable to open serial port "'
                +SerPortname
                +'".'
                +LineEnding
                +'Check name and user privileges.'
                + LineEnding
                +'Retry?',
                mtConfirmation, [mbYES, mbNO],0) = mrNO then begin
                CloseProgram
            end else begin
                In_Command := FALSE;
            end;
        end
    end;
    In_Command := FALSE;
end;

// BFO
procedure TRA1792.BBFOClick(Sender: TObject);
begin
    if KeyboardState = STATE_NORMAL then begin
        KeyboardState := STATE_SETBFO;
        LBBFO.Visible := TRUE;
        RXState.TUNE := FALSE;
        LBTUNE.Visible := FALSE;
        Message(SETBFOFREQ);
    end else if KeyboardState = STATE_SETBFO then begin
        KeyboardState := STATE_NORMAL;
        LBBFO.Visible := FALSE;
    end;
    SetActiveButtons;
end;

// TUNE
procedure TRA1792.BTUNEClick(Sender: TObject);
begin
    if keyboardState = STATE_NORMAL then begin
        RXState.Tune := not RXState.Tune;
        if RXState.Tune then begin
            LBTUNE.Visible := TRUE;
            SPFREQ.Enabled := TRUE;
        end else begin
          LBTUNE.Visible := FALSE;
          SPFREQ.Enabled := FALSE;
        end;
        LBCHAN.Visible := FALSE;
        CHANTENS.Visible := FALSE;
        CHANUNITS.Visible := FALSE;
    end;
end;

//
// Buttons handling (block 2)
//


// "1/BW1"
procedure TRA1792.B1Click(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            while not SendCommand(Set_Bandwidth, IntToStr(Filter_300)) do;
            RXState.Filter := FILTER_300;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('1');
        end;
        STATE_SCAN: begin
            ScanDelay := 2*Delay100ms;
            Message(SCANDELAYMSG+' 0.2s');
        end;
    end;
    In_Command := FALSE;
end;

// "2/BW2"
procedure TRA1792.B2Click(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            while not SendCommand(Set_Bandwidth, IntToStr(Filter_1000)) do;
            RXState.Filter := FILTER_1000;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('2');
        end;
        STATE_SCAN: begin
            ScanDelay := 5*Delay100ms;
            Message(SCANDELAYMSG+' 0.5s');
        end;
    end;
    In_Command := FALSE;
end;

// "3/BW3"
procedure TRA1792.B3Click(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            while not SendCommand(Set_Bandwidth, IntToStr(Filter_3200)) do;
            RXState.Filter := FILTER_3200;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('3');
        end;
        STATE_SCAN: begin
            ScanDelay := 8*Delay100ms;
            Message(SCANDELAYMSG+' 0.8s');
        end;
    end;
    In_Command := FALSE;
end;

// "4/BW4"
procedure TRA1792.B4Click(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            while not SendCommand(Set_Bandwidth, IntToStr(Filter_6000)) do;
            RXState.Filter := FILTER_6000;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('4');
        end;
        STATE_SCAN: begin
            ScanDelay := 10*Delay100ms;
            Message(SCANDELAYMSG+' 1s');
        end;
    end;
    In_Command := FALSE;
end;

// "5/BW5"
procedure TRA1792.B5Click(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            while not SendCommand(Set_Bandwidth, IntToStr(Filter_16000)) do;
            RXState.Filter := FILTER_16000;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('5');
        end;
        STATE_SCAN: begin
            ScanDelay := 20*Delay100ms;
            Message(SCANDELAYMSG+' 2s');
        end;
    end;
    In_Command := FALSE;
end;

// "6/MAN"
procedure TRA1792.B6Click(Sender: TObject);

var spar: string;
    ipar: integer = 0;

begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            case RXState.AGC of
                AGC_SHORT, AGC_MEDIUM, AGC_LONG:
                  ipar := RXState.AGC+AGC_MAN;
                AGC_MAN_SHORT, AGC_MAN_MEDIUM, AGC_MAN_LONG:
                  ipar := RXState.AGC-AGC_MAN;
                AGC_MAN: ipar := AGC_MAN;
            end;
            spar := IntToStr(ipar);
            RXState.AGC := ipar;
            while not SendCommand(Set_AGC, spar) do;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('6');
        end;
        STATE_SCAN: begin
            ScanDelay := 40*Delay100ms;
            Message(SCANDELAYMSG+' 4s');
        end;
    end;
    In_Command := FALSE;
end;

// "7/SHORT"
procedure TRA1792.B7Click(Sender: TObject);

var spar: string;
    ipar: integer = 0;

begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            case RXState.AGC of
                AGC_SHORT, AGC_MEDIUM, AGC_LONG: ipar := AGC_SHORT;
                AGC_MAN_SHORT: ipar := AGC_MAN;
                AGC_MAN_MEDIUM, AGC_MAN_LONG: ipar := AGC_MAN + AGC_SHORT;
                AGC_MAN: ipar := AGC_MAN+AGC_SHORT;
            end;
            spar := IntToStr(ipar);
            RXState.AGC := ipar;
            while not SendCommand(Set_AGC, spar) do;
            UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('7');
        end;
        STATE_SCAN: begin
            ScanDelay := 60*Delay100ms;
            Message(SCANDELAYMSG+' 6s');
        end;
    end;
    In_Command := FALSE;
end;

// "8/MED"
procedure TRA1792.B8Click(Sender: TObject);

var spar: string;
    ipar: integer = 0;

begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
            case RXState.AGC of
               AGC_SHORT, AGC_MEDIUM, AGC_LONG: ipar := AGC_MEDIUM;
               AGC_MAN_SHORT, AGC_MAN_LONG: ipar := AGC_MAN + AGC_MEDIUM;
               AGC_MAN_MEDIUM: ipar := AGC_MAN;
               AGC_MAN: ipar := AGC_MAN+AGC_MEDIUM;
           end;
           spar := IntToStr(ipar);
           RXState.AGC := ipar;
           while not SendCommand(Set_AGC, spar) do;
           UpdateDisplay;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('8');
        end;
        STATE_SCAN: begin
            ScanDelay := 80*Delay100ms;
            Message(SCANDELAYMSG+' 8s');
        end;
    end;
    In_Command := FALSE;
end;

// "9/LONG"
procedure TRA1792.B9Click(Sender: TObject);

var spar: string;
    ipar: integer = 0;

begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
           case RXState.AGC of
              AGC_SHORT, AGC_MEDIUM, AGC_LONG: ipar := AGC_LONG;
              AGC_MAN_SHORT, AGC_MAN_MEDIUM: ipar := AGC_MAN + AGC_LONG;
              AGC_MAN_LONG: ipar := AGC_MAN;
              AGC_MAN: ipar := AGC_MAN+AGC_LONG;
          end;
          spar := IntToStr(ipar);
          RXState.AGC := ipar;
          while not SendCommand(Set_AGC, spar) do;
          UpdateDisplay;
      end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('9');
        end;
        STATE_SCAN: begin
            ScanDelay := 100*Delay100ms;
            Message(SCANDELAYMSG+' 10s');
        end;
    end;
    In_Command := FALSE;
end;

// "Store"
procedure TRA1792.BSTOREClick(Sender: TObject);
begin
    case KeyboardState of
        STATE_NORMAL: begin
            KeyboardState := STATE_STOCHAN;
            LBCHAN.Visible := TRUE;
            CHANTENS.Visible := TRUE;
            CHANUNITS.Visible := TRUE;
            iCHNum := 1;
            sCHTmp := Format('%2.2D',[CurChan]);
            PutChanString(sCHTmp);
            SetKeyCaptions(KBMODE_NUMKEYS);
        end;
        STATE_STOCHAN: begin
            BEnterClick(Sender);
        end;
    end;
end;

// "0/AUX"
procedure TRA1792.B0Click(Sender: TObject);

var tmpRXState: TRXState;
    AUXFile: file of TRXstate;

begin
    if In_Command then exit;
    In_Command := TRUE;
    case KeyboardState of
        STATE_NORMAL: begin
          AssignFile(AUXFile, Homedir+'States/'+AUXFileName);
          try
              Reset(AUXFile);
          except
              Message(NOAUXFILE);
              In_Command := FALSE;
              exit;
          end;
          Read(AUXFile,tmpRXState);
          CloseFile(AUXFile);
          RXState.Mode_RX := tmpRXState.Mode_RX;
          RXstate.Filter := tmpRXState.Filter;
          RXstate.AGC := tmpRXState.AGC;
          RXState.BFO := tmpRXState.BFO;
          RestoreState;
        end;
        STATE_SETFREQ, STATE_SETBFO, STATE_SELCHAN, STATE_STOCHAN: begin
            KeyboardHandler('0');
        end;
        STATE_SCAN: begin
            ScanDelay := Delay100ms;
            Message(SCANDELAYMSG+' 0.1s');
        end;
    end;
    In_Command := FALSE;
end;

// "ENTER"
procedure TRA1792.BENTERClick(Sender: TObject);

var p, chnum: integer;
    s: string;

begin
    In_Command := TRUE;
    case KeyboardState of
        STATE_SETFREQ: begin
            s := GetRXString;
            p := pos('-',s);
            if p <> 0 then s[p] := '0';
            RXState.Freq_rx := Trunc(1000*StrToFloat(AdjustDecimalSeparator(s)));
            SetRXfreq;
            SetKeyCaptions(KBMODE_FUNKEYS);
            KeyboardState := STATE_NORMAL;
       end;
        STATE_SELCHAN: begin
            chnum := StrToInt(GetChanString);
            RXstate := RXChannels[chnum];
            if not ((RXstate.Freq_step in [1,10,100]) or (RXstate.Freq_step=1000)) then
                RXState.Freq_step := 100;
            RXState.Tune := FALSE;
            LastChan := chnum;
            RestoreState;
            LBCHAN.Visible := FALSE;
            CHANTENS.Visible := FALSE;
            CHANUNITS.Visible := FALSE;
            SetKeyCaptions(KBMODE_FUNKEYS);
            KeyboardState := STATE_NORMAL;
        end;
        STATE_STOCHAN: begin
          chnum := StrToInt(GetChanString);
          RXChannels[chnum] := RXState;
          LastChan := chnum;
          rCurChan := chnum;
          KeyboardState := STATE_SELCHAN;
        end;
        STATE_BITE: begin
          GetStatus;
          UpdateDisplay;
        end;
    end;
    UpdateDisplay;
    SetActiveButtons;
    In_Command := FALSE;
end;

// FREQ
procedure TRA1792.BFREQClick(Sender: TObject);

begin
    SetKeyCaptions(KBMODE_NUMKEYS);
    KeyboardState := STATE_SETFREQ;
    STIND.Visible := TRUE;
    RXState.Tune := FALSE;
    LBTUNE.Visible := FALSE;
    SetActiveButtons;
    PutRXString('-0000.000');
    Message(ENTERFREQKHZ);
end;

// CHAN
procedure TRA1792.BCHANClick(Sender: TObject);
begin
    KeyboardState := STATE_SELCHAN;
    LBCHAN.Visible := TRUE;
    CHANTENS.Visible := TRUE;
    CHANUNITS.Visible := TRUE;
    STIND.Visible := TRUE;
    LBTUNE.Visible := FALSE;
    OldRXState := RXState;
    RXState := RXChannels[LastChan];
    SetKeyCaptions(KBMODE_NUMKEYS);
    SetActiveButtons;
    UpdateDisplay;
    iCHNum := 1;
    sCHTmp := IntToStr(LastChan);
    rCurChan := LastChan;
    if Length(sCHTmp) = 1 then sCHtmp := '0'+sCHTmp;
    Message(ENTERCHANNO);
end;

// RCL
procedure TRA1792.BRCLClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    RXState := OldRXState;
    KeyboardState := STATE_NORMAL;
    SetKeyCaptions(KBMODE_FUNKEYS);
    LBCHAN.Visible := FALSE;
    CHANTENS.Visible := FALSE;
    CHANUNITS.Visible := FALSE;
    RXstate.Tune := FALSE;
    RestoreState;
    SetActiveButtons;
    UpdateDisplay;
    In_Command := FALSE;
end;

//
// Buttons handling (block 3)
//

// "ISB"
procedure TRA1792.BISBClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_ISB)) do;
    RXState.Mode_RX := M_ISB;
    GetStatus;
    SetActiveButtons;
    UpdateDisplay;
    In_Command := FALSE;
end;

// "LSB"
procedure TRA1792.BLSBClick(Sender: TObject);

begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_LSB)) do;
        RXState.Mode_RX := M_LSB;
    SetActiveButtons;
    GetStatus;
    UpdateDisplay;
    In_Command := FALSE;
end;

// "USB"
procedure TRA1792.BUSBClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_USB)) do;
    RXState.Mode_RX := M_USB;
    SetActiveButtons;
    GetStatus;
    UpdateDisplay;
    In_Command := FALSE;
end;

// "AM"
procedure TRA1792.BAMClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_AM)) do;
    RXState.Mode_RX := M_AM;
    SetActiveButtons;
    GetStatus;
    UpdateDisplay;
    In_Command := FALSE;
end;

// "CW"
procedure TRA1792.BCWClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_CW)) do;
    RXState.Mode_RX := M_CW;
    GetStatus;
    SetActiveButtons;
    UpdateDisplay;
    In_Command := FALSE;
end;

// "FM"
procedure TRA1792.BFMClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(M_FM)) do;
    RXState.Mode_RX := M_FM;
    SetActiveButtons;
    GetStatus;
    UpdateDisplay;
    In_Command := FALSE;
end;

//
// Buttons handling (block 4)
//

// "CHAN SCAN"
procedure TRA1792.BCHSCANClick(Sender: TObject);

var i: integer;
    t: TTime;
    SomeChanScanned: boolean = FALSE;

begin
    case KeyboardState of
        STATE_NORMAL: begin
          KeyboardState := STATE_SCAN;
          SetActiveButtons;
          SetKeyCaptions(KBMODE_SCANDELAY);
          OldRXState := RXState;
          i := 0;
          t := Time;
          LBSCAN.Visible := TRUE;
          while KeyBoardState = STATE_SCAN do begin
            if RXChannels[i].ScanFlag then begin
                SomeChanScanned := TRUE;
                RXState := RXChannels[i];
                RestoreState;
                UpdateDisplay;
                t := Time+ScanDelay;
            end;
            while Time < t do DoMessages;
            i := i + 1;
            if i > 99 then begin
                if SomeChanScanned then
                    i := 0
                else begin
                    Message(NOCHANSCANF);
                    BCHSCANClick(Sender);
                end;
            end;
          end;
        end;
        STATE_SCAN: begin
          KeyboardState := STATE_NORMAL;
          LBSCAN.Visible := FALSE;
          SetActiveButtons;
          SetKeyCaptions(KBMODE_FUNKEYS);
        end;
        STATE_SELCHAN: begin
           RXChannels[CurChan].ScanFlag := not RXChannels[CurChan].ScanFlag;
           if RXChannels[CurChan].ScanFlag then
               LBSCAN.Visible := TRUE
           else
               LBSCAN.Visible := FALSE;
        end;
    end;
end;


//
// ComboBox & SpinEdit
//

// AGC ComboBox
procedure TRA1792.CBXAGCChange(Sender: TObject);
begin
    In_Command := TRUE;
    while not SendCommand(Set_AGC, IntToStr(CBXAGC.ItemIndex+1)) do;
    RXstate.AGC := CBXAGC.ItemIndex+1;
    EnableCmdKeys;
    EnableCombosAndEdits;
    UpdateDisplay;
    In_Command := FALSE;
end;

// BANDWIDTH ComboBox
procedure TRA1792.CBXBWChange(Sender: TObject);
begin
    In_Command := TRUE;
    while not SendCommand(Set_Bandwidth, IntToStr(CBXBW.ItemIndex+1)) do;
    EnableCmdKeys;
    EnableCombosAndEdits;
    RXState.Filter := CBXBW.ItemIndex+1;
    GetStatus;
    UpdateDisplay;
    In_Command := FALSE;
end;

// MODE ComboBox
procedure TRA1792.CBXMODEChange(Sender: TObject);

begin
    In_Command := TRUE;
    while not SendCommand(Set_Mode, IntToStr(CBXMODE.ItemIndex+1)) do;
    RXstate.Mode_RX := CBXMODE.ItemIndex+1;
    GetStatus;
    UpdateDisplay;
    EnableCmdKeys;
    EnableCombosAndEdits;
    SetActiveButtons;
    In_Command := FALSE;
end;

procedure TRA1792.CBXMODEMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
begin

end;

// Frequency UP/DOWN
procedure TRA1792.SPFREQChange(Sender: TObject);
begin
    {$IFDEF LINUX}
    if Update_Only or In_Command or (not RXState.Tune) then exit;
    In_Command := TRUE;
    {$ENDIF}
    {$IFDEF WINDOWS}
    // Under Windows the OnChange event fires also when it shouldnt
    if Update_Only or (RXState.Freq_RX = SPFREQ.Value) then exit;
    {$ENDIF}
    RXState.Freq_RX := SPFREQ.Value;
    SetRXFreq;
    UpdateDisplay;
    In_Command := FALSE;
end;

procedure TRA1792.SPFREQMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

begin
    If WheelDelta > 0 then SPFREQ.Value := SPFREQ.Value + 1;
    If WheelDelta < 0 then SPFREQ.Value := SPFREQ.Value - 1;
    Handled := TRUE;
end;

// Attenuation slider
procedure TRA1792.TBATTENChange(Sender: TObject);

var s: string;

begin
    s := IntToStr(TBATTEN.Position);
    while Length(s) < 3 do s := '0'+s;
    SendCommand(Set_Attenuation, s);
    {$IFDEF TEST_RFGAIN}
    itmp := (12*TBatten.Position) div 150;
    s := Copy(MeterChars, 1, 2*itmp);
    LBMET.Caption := s;
    {$ENDIF}
end;

// Alternative frequency setting way: left/right click on the frequency figures
// Clicking on the decimal point rewrites displayed frequency
procedure TRA1792.RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);

var step: integer;

begin
    if In_Command or In_RXFreqMouseUp or (not RXstate.Tune) then exit;

    {$IFDEF TEST_FREQMOUSEUP}
    Message('X='+IntToStr(X)+' Y='+IntTostr(Y));
    {$ENDIF}

    In_RXFreqMouseUp := TRUE;
    step := (Sender as TLabel).tag;
    if step < 0 then step := 0;

    if Button = mbLeft then begin
        if RXState.Freq_rx - step >= MINRXF then
	   RXState.Freq_rx := RXState.Freq_rx-step;
	end;
    if Button = mbRight then begin
        if RXState.Freq_rx + step <= MAXRXF then
	   RXState.Freq_rx := RXState.Freq_rx+step;
	end;
    if not SetRXFreq then
        Message(COMMANDFAILED);
    DispRXf;

    In_RXFreqMouseUp := FALSE;
end;

// Wheel over frequency display
procedure TRA1792.RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

    var step: integer = -1;

begin
    if In_Wheel or In_Status or (not RXState.Tune) then exit;
    In_Wheel := TRUE;
    {$IFDEF TEST_FREQMOUSEWHEEL}
    Message('X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y));
    {$ENDIF}

    step := (Sender as TLabel).Tag;
    if step < 0 then step := 0;

    // Mouse wheel over RX frequency display
    if WheelDelta > 0 then begin
        if step >= 0 then begin
	        if RXState.Freq_rx + step <= MAXRXF then begin
                RXState.Freq_rx := RXState.Freq_rx + step;
                if not SetRXFreq then Message(COMMANDFAILED);
            end else Message(RXCANTTUNE);
        end;
    end;
    if WheelDelta < 0 then begin
        if step >= 0 then begin
	        if RXState.Freq_rx - step >= MINRXF then begin
                RXState.Freq_rx := RXState.Freq_rx-step;
                if not SetRXFreq then Message(COMMANDFAILED);
            end else Message(RXCANTTUNE);
        end;
    end;
    UpdateDisplay;
    SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);

    Handled := TRUE;
    In_Wheel := FALSE;
end;

// Mouse selection of channel number

// Mouse click on channel display: left=DOWN, right=UP
procedure TRA1792.CHANMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);

var s: string;
    delta: integer;

begin
    if In_Command or In_ChanMouseUp then exit;
    In_ChanMouseUp := TRUE;
    delta := (Sender as TLabel).Tag;
    if Button = MBLEFT then CurChan := CurCHan-delta;
    if Button = MBRIGHT then CurChan := CurChan+delta;
    if CurChan > 99 then CurChan := CurChan-100;
    if CurChan < 0  then CurChan := CurChan+100;

    s := Format('%2.2D',[CurChan]);
    PutChanString(s);
    LastChan := CurChan;
    if KeyboardState = STATE_SELCHAN then
        RXState := RXChannels[CurChan];
    UpdateDisplay;
    In_ChanMouseUp := FALSE;
end;

// Mouse Wheel on channel display
procedure TRA1792.CHANMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var s: string;
    delta: integer;

begin
    if In_Command or In_ChanMouseWheel then exit;

    {$IFDEF TEST_CHANMOUSEWHEEL}
    Message('X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y));
    {$ENDIF}

    In_ChanMouseWheel := TRUE;

    delta := (Sender as TLabel).Tag;

    if (WheelDelta>0) then CurChan := (CurChan+delta);
    if (WheelDelta<0) then CurChan := (CurChan-delta);
    if CurChan > 99 then CurChan := CurChan-100;
    if CurChan < 0 then  CurChan := CurChan+100;
    s := Format('%2.2D',[CurChan]);
    PutChanString(s);
    LastChan := CurChan;
    if KeyboardState = STATE_SELCHAN then
        RXState := RXChannels[CurChan];
    UpdateDisplay;
    In_ChanMouseWheel := FALSE;
    Handled := TRUE;
end;


// Third tuning mode: simulated tuning knob
procedure TRA1792.SHKNOB1MouseMove(Sender: TObject; Shift: TShiftState; X,
    Y: Integer);

var alpha, dalpha, myx, myy, deltatms: real;
    tmpstep,step,direction: integer;
    s: string;
    t: TTime;

begin
    if In_Command or not (RXstate.Tune or
       (KeyboardState in [STATE_SETBFO..STATE_STOCHAN])) then exit;

    t := Time;
    deltatms := (t-LastTuningTime)/OneMillisecond;
    LastTuningTime := t;

    {$IFDEF TEST_KNOBMOUSEMOVE}
    Message('deltat = '+IntToStr(trunc(deltatms))+ ' ms');
    {$ENDIF}

    if deltatms>1000 then deltatms := 1000;

    {$IFDEF TEST_KNOBMOUSEMOVE}
    Message('step = '+IntToStr(step)+ ' Hz');
    {$ENDIF}

    if (sqr(X-X2)+sqr(Y-Y2)) > sqr(R2) then exit;

    In_Command := TRUE;

    Y := Y+KnobOffsetY;
    X := X+KnobOffsetX;
    alpha := arctan2(Y-SHKNOB1.Height / 2, X - SHKNOB1.Width / 2);
    myx := r * cos(alpha) + cx;
    myy := r * sin(alpha) + cy;
    SHKNOB.Top := trunc(myy);
    SHKNOB.Left := trunc(myx);
    {$IFDEF TEST_KNOBMOUSEMOVE}
    Message('alpha='+floatTostr(alpha)+' prevalpha='+floatTostr(prevalpha));
    {$ENDIF}
    alpha := alpha+pi; // make angle interval positive (0..2*pi)
    dalpha := alpha-prevalpha;
    if (dalpha) < -pi then begin
        prevalpha := 0;   // (2*pi) --> 0
        dalpha := alpha-prevalpha;
    end;
    if (dalpha) > pi then begin
        prevalpha := 2*pi; // 0 --> 2*pi
        dalpha := alpha-prevalpha;
    end;
    if dalpha > 0 then direction := 1 else direction := -1;
    tmpstep := trunc(HzPerTurn*((dalpha)/(2*pi)));
    prevalpha := alpha;

    if MTIMED.Checked then begin
//        step := trunc((301-deltatms)*direction);
          case trunc(deltatms) of
              0..25:   step := 10000;
              26..200: step := 1000;
              201..500: step := 100;
              501..800: step := 10;
              801..1000: step := 1;
          end;
          step := step * direction;
    end else begin
        step := tmpstep;
    end;
    case KeyboardState of
        STATE_NORMAL: begin
            RXState.Freq_RX := RXState.Freq_rx+step;
            if RXState.Freq_RX > MAXRXF then begin
                RXState.Freq_RX := MAXRXF;
                Message(RXCANTTUNE);
            end;
            if RXState.Freq_RX < MINRXF then begin
                RXState.Freq_RX := MINRXF;
                Message(RXCANTTUNE);
            end;
            SetRXFreq;
            DispRXF;
        end;
        STATE_SETBFO: begin
            if step > 0 then step := 10 else step := -10;
            RXstate.BFO := RXstate.BFO + step;
            if RXstate.BFO > 8000 then RXState.BFO := 8000;
            if RXstate.BFO < -8000 then RXState.BFO := -8000;
            Str(RXState.BFO/1000:4:2, s);
            SendCommand(Set_BFO, s);
            UpdateDisplay;
        end;
        STATE_SELCHAN, STATE_STOCHAN: begin
            if step > 0 then
               rCurChan := rCurChan + 0.2
            else
               rCurChan := rCurChan - 0.2;
            if rCurChan < 0 then rCurChan := 99;
            if rCurChan > 99 then rCurChan := 0;
            CurChan := trunc(rCurChan);
            s := Format('%2.2D',[CurChan]);
            PutChanString(s);
            LastChan := CurChan;
            if KeyboardState = STATE_SELCHAN then
                RXState := RXChannels[CurChan];
            UpdateDisplay;
        end;
    end;
    In_Command := FALSE;
end;

// Click on the tunign knob hole
procedure TRA1792.SHKNOBMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
begin
    case KeyboardState of
        STATE_NORMAL: begin
            RXState.Tune := not RXState.Tune;
            if RXstate.Tune then begin
             SPFREQ.Enabled := TRUE;
             LastTuningTime := Time;
            end else
            SPFREQ.Enabled := FALSE;
            UpdateDisplay;
        end;
        STATE_SELCHAN: BENTERClick(Sender);
        STATE_STOCHAN: BENTERClick(Sender);
        STATE_SETBFO: BBFOCLick(Sender);
    end;
end;

// Allows to use mouse wheel for tuning
procedure TRA1792.FormMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var s: string;

begin
    if In_Wheel or In_Status then exit;
    In_Wheel := TRUE;
    {$IFDEF TEST_FORMMOUSEWHEEL}
    Message('X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y));
    {$ENDIF}
    if (KeyboardState=STATE_NORMAL) and RXstate.Tune then begin
        if WheelDelta > 0 then begin
            if RXState.Freq_rx+RXState.Freq_step <= MAXRXF then begin
                RXState.Freq_rx := RXState.Freq_rx + RXState.Freq_step;
                if SetRXFreq then begin
                    GetStatus;
                    ParseStatus(status);
                    UpdateDisplay;
                end else Message(COMMANDFAILED);
            end else Message(RXCANTTUNE);
        end;
        if WheelDelta < 0 then begin
            if RXState.Freq_rx-RXState.Freq_step >= MINRXF then begin
                RXState.Freq_rx := RXState.Freq_rx - RXState.Freq_step;
                if SetRXFreq then begin
                    GetStatus;
                    ParseStatus(status);
                    UpdateDisplay;
                end else Message(COMMANDFAILED);
            end else Message(RXCANTTUNE);
        end;
    end;
    if (KeyBoardState=STATE_SELCHAN) or (KeyBoardState=STATE_STOCHAN) then begin
        if (WheelDelta>0) then CurChan := (CurChan+1);
        if (WheelDelta<0) then CurChan := (CurChan-1);
        if CurChan < 0 then CurChan := 99;
        if CurChan > 99 then CurChan := 0;
        s := Format('%2.2D',[CurChan]);
        PutChanString(s);
        if KeyBoardState=State_SELCHAN then begin
            if KeyboardState = STATE_SELCHAN then
                RXState := RXChannels[CurChan];
            UpdateDisplay;
        end;
        LastChan := CurChan;
    end;
    Handled := TRUE;
    In_Wheel := FALSE;
end;

//
// Timers
//

// Status timer. Get signal level every 300 ms.
// May not work, my RA1792 does not have the level reading command.
// Level is returned also in response to G command or in TR command only?
procedure TRA1792.TSmeterTimer(Sender: TObject);

begin
    if Enable_Status and (KeyboardState <> STATE_POWEROFF) then begin
        if In_Status or In_Command then exit;
        In_Status := TRUE;
        status := '';
        GetLevel;
        In_Status := FALSE;
    end;
end;

// Display signal level every 300 ms
// Manual says RF level is 0..150, why not 0..120?
procedure TRA1792.TSMDISPTimer(Sender: TObject);

var itmp: integer;
    s: string;

begin
  {$IFDEF TEST_USER_INTERFACE}
  SignalLevel := 100+random(50);
  {$ENDIF}
  itmp := (12*SignalLevel) div 150;
  s := Copy(MeterChars, 1, 2*itmp);
  LBMET.Caption := s;
end;

// Timer to update main window if required
procedure TRA1792.UPDDISPTimer(Sender: TObject);
begin
  // Update displayed data
  if PanelUpdateRequired then begin
    PanelUpdateRequired := FALSE;
    UpdateDisplay;
  end;

  // Message to show in the message window
  if MsgToShow<>'' then begin
    Message(MsgToShow);
    MsgToShow := '';
  end;
end;


// Things to do at startup
initialization

begin
  // Use internal Lazarus versioning
  if GetProgramVersion(ProgVersion) then begin
    Major := ProgVersion[1];
    Minor := ProgVersion[2];
    Revision := ProgVersion[3];
    Build := ProgVersion[4];
    // Make version strings
    ShortVersion := 'v'+IntTostr(Major)+'.'+IntTostr(Minor)+IntToStr(revision)+
                 ' build '+IntToStr(Build);
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end else begin
    ShowMessage(GETPROGVFAIL);
    ShortVersion := NOVERSINFO;
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end;

  {$IFDEF LINUX}
  HomeDir := GetUserDir+'.RA1792Control/';
  StateDir := HomeDir+'States/';
  ChannelsDir := HomeDir+'Channels/';
  BandScansdir := Homedir+'BandScans/';
  DocDir := '/usr/share/doc/ra1792control/';
  GPLv3Dir := '/usr/share/doc/ra1792control/GPLv3/';
  {$ENDIF}
  {$IFDEF WINDOWS}
  // Use the executable path to discover the location of the
  // RA1792Control directory.
  HomeDir := ExtractFilePath(Application.ExeName);
  StateDir := HomeDir+'States\';
  ChannelsDir := HomeDir+'Channels\';
  BandScansdir := Homedir+'BandScans\';
  DocDir := HomeDir+'doc\';
  GPLv3Dir := HomeDir+'GPLv3\';
  {$ENDIF}
  if not DirectoryExists(HomeDir) then CreateDir(HomeDir);
  if not DirectoryExists(StateDir) then CreateDir(StateDir);
  if not DirectoryExists(BandsDir) then CreateDir(BandsDir);
  if not DirectoryExists(BandScansDir) then CreateDir(BandScansDir);
  if not DirectoryExists(ChannelsDir) then CreateDir(ChannelsDir);

  // Check manual and license files
  if not (FileExists(DocDir+'Manual-en.txt') and
          FileExists(GPLv3Dir+'gpl-3.0.txt')) then
    ErrMsg := DOCFILESNOTF+LineEnding+CHECKINST;

  // If installation is severely broken, notify user and close program now.
  if ErrMsg <> '' then begin
    MessageDlg(ERRORMSG, ErrMsg, mtError,[mbOK],0);
    halt;
  end;
end;

end.

