{
    This file is part of EKD500Control

    RFT EKD500 Control program

    Copyright (C) 2012-2022 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/.

                                    * * *

    Hamlib rigctld-maybe-compatible EKD500 control net server.
    The strictly-networking part is in ekd500.pas using lNET.

    Rigctld commands are only partially coded. The implementation uses
    only the basic remote command capabilities of the EKD500.
    ==========================================================================
                                   WANTED
    ==========================================================================
             Someone with better knowledge of how rigctld works
                to help complete/correct the implementation.
    ==========================================================================
}

//
// Hamlib command parser
//

{$include defines.inc}

unit UParse;

{$mode ObjFPC}

interface

uses
  Classes, SysUtils, UState, StrUtils, lNet;

procedure ParseNetMsg(RXLine: string; MySocket: TLSocket);

// Messages declared here for i18n
resourcestring
  REMCMDNOTREC='Remote command not recognized: %s';
  REMCMDNOTIMP='Remote command not implemented: %s';
  REMCMDNOTAVA='Remote command not available: %s';
  EXTPROTNOTSUPP='Extended response protocol is not supported';

implementation

uses ekd500;

type

  // The hamlib rigtcld commands
  TCmdCodes =
    (SET_FREQ, GET_FREQ, SET_MODE, GET_MODE, SET_VFO, GET_VFO,
     SET_RIT, GET_RIT, SET_XIT, GET_XIT, SET_PTT, GET_PTT,
     SET_SPLIT_VFO, GET_SPLIT_VFO, SET_SPLIT_FREQ, GET_SPLIT_FREQ,
     SET_SPLIT_MODE, GET_SPLIT_MODE, SET_ANT, GET_ANT, SEND_MORSE,
     GET_DCD, SET_RPTR_SHIFT, GET_RPTR_SHIFT, SET_RPTR_OFFS,GET_RPTR_OFFS,
     SET_CTCSS_TONE, GET_CTCSS_TONE, SET_DCS_CODE, GET_DCS_CODE,
     SET_CTCSS_SQL, GET_CTCSS_SQL, SET_DCS_SQL, GET_DCS_SQL, SET_TS,
     GET_TS, SET_FUNC, GET_FUNC, SET_LEVEL, GET_LEVEL, SET_PARM, GET_PARM,
     SET_BANK, SET_MEM, GET_MEM, VFO_OP, SCAN, SET_CHANNEL, GET_CHANNEL,
     SET_TRN, GET_TRN, RESET, SET_POWERSTAT, GET_POWERSTAT, SEND_DTMF,
     RECV_DTMF, GET_INFO, GET_RIG_INFO, GET_VFO_INFO, DUMP_STATE,DUMP_CAPS,
     POWER2MW, MW2POWER, SET_CLOCK, GET_CLOCK, CHK_VFO, SET_VFO_OPT, QUIT,
     LASTCMD
  );

  // The hamlib rigtcld VFOs
  TVFONames =
    (NONE, VFOA, VFOB, VFOC, currVFO, VFO, MEM, Main, Sub, TX, RX);

  // The hamlib rigtcld levels
  TLevels = (PREAMP,ATT,LVOX,AF,LRF,LSQL,LIF,LAPF,LNR,PBT_IN,PBT_OUT,CWPITCH,RFPOWER,
     RFPOWER_METER,RFPOWER_METER_WATTS,MICGAIN,KEYSPD,NOTCHF,LCOMP,AGC,BKINDL,BAL,
     METER,VOXGAIN,ANTIVOX,SLOPE_LOW,SLOPE_HIGH,RAWSTR,SWR,ALC,STRENGTH,
     LASTLEVEL);

  // The hamlib rigtcld functions
  TFunctions =
    (FAGC,NB,COMP,FVOX,TONE,TSQL,SBKIN,FBKIN,ANF,NR,AIP,APF,MON,MN,FRF,ARO,LOCK,
     MUTE,VSC,REV,FSQL,ABM,BC,MBC,RIT,AFC,SATMODE,SCOPE,RESUME,TBURST,TUNER,XIT,
     LASTFUNCTION
    );

  // Hamlib error codes
  // Taken from hamlib sources (rig.h)
  THamlibErrors = (
    RIG_OK,           //  0 No error, operation completed successfully */
    RIG_EINVAL,       //  1 invalid parameter */
    RIG_ECONF,        //  2 invalid configuration (serial,..) */
    RIG_ENOMEM,       //  3 memory shortage */
    RIG_ENIMPL,       //  4 function not implemented, but will be */
    RIG_ETIMEOUT,     //  5 communication timed out */
    RIG_EIO,          //  6 IO error, including open failed */
    RIG_EINTERNAL,    //  7 Internal Hamlib error, huh! */
    RIG_EPROTO,       //  8 Protocol error */
    RIG_ERJCTED,      //  9 Command rejected by the rig */
    RIG_ETRUNC,       // 10 Command performed, but arg truncated */
    RIG_ENAVAIL,      // 11 Function not available */
    RIG_ENTARGET,     // 12 VFO not targetable */
    RIG_BUSERROR,     // 13 Error talking on the bus */
    RIG_BUSBUSY,      // 14 Collision on the bus */
    RIG_EARG,         // 15 NULL RIG handle or any invalid pointer parameter in
    RIG_EVFO,         // 16 Invalid VFO */
    RIG_EDOM,         // 17 Argument out of domain of func */
    RIG_EDEPRECATED,  // 18 Function deprecated */
    NOTREQUIRED = 99  // 99 Was a get command, no RPRT x reply required
  );

  // The command parser structure
  THamLibCmd = record
    LongName: string;
    ShortName: string;
    CmdCode: TCmdCodes;
  end;

  // The level parser structure
  THamlibLevel = record
    LevelName: string;
    LevelCode: TLevels;
  end;

  // The function parser structure
  THamlibFunction = record
    FunctionName: string;
    FunctionCode: TFunctions;
  end;

  // The commands array
  THamlibCmds = array[1..ord(LASTCMD)] of THamLibCmd;

  // The levels array
  THamlibLevels = array[1..ord(LASTLEVEL)] of THamlibLevel;

  // The functions array
  THamlibFunctions = array[1..ord(LASTFUNCTION)] of THamlibFunction;

  // Various constant values
  const
    RPRT: string = 'RPRT ';
    RPRT_OK: string = 'RPRT 0';
    VFOMode: integer = 0;

  HamLibCmds: THamlibCmds = (
    (LongName:'set_freq';ShortName:'F';CmdCode:SET_FREQ),
    (LongName:'get_freq';ShortName:'f';CmdCode:GET_FREQ),
    (LongName:'set_mode';ShortName:'M';CmdCode:SET_MODE),
    (LongName:'get_mode';ShortName:'m';CmdCode:GET_MODE),
    (LongName:'set_vfo';ShortName:'V';CmdCode:SET_VFO),
    (LongName:'get_vfo';ShortName:'v';CmdCode:GET_VFO),
    (LongName:'set_rit';ShortName:'J';CmdCode:SET_RIT),
    (LongName:'get_rit';ShortName:'j';CmdCode:GET_RIT),
    (LongName:'set_xit';ShortName:'Z';CmdCode:SET_XIT),
    (LongName:'het_xit';ShortName:'z';CmdCode:GET_XIT),
    (LongName:'set_ptt';ShortName:'T';CmdCode:SET_PTT),
    (LongName:'get_ptt';ShortName:'t';CmdCode:GET_PTT),
    (LongName:'set_split_vfo';ShortName:'S';CmdCode:SET_SPLIT_VFO),
    (LongName:'get_split_vfo';ShortName:'s';CmdCode:GET_SPLIT_VFO),
    (LongName:'set_split_freq';ShortName:'I';CmdCode:SET_SPLIT_FREQ),
    (LongName:'get_split_freq';ShortName:'i';CmdCode:GET_SPLIT_FREQ),
    (LongName:'set_split_mode';ShortName:'X';CmdCode:SET_SPLIT_MODE),
    (LongName:'get_split_mode';ShortName:'x';CmdCode:GET_SPLIT_MODE),
    (LongName:'set_ant';ShortName:'Y';CmdCode:SET_ANT),
    (LongName:'get_ant';ShortName:'y';CmdCode:GET_ANT),
    (LongName:'send_morse';ShortName:'b';CmdCode:SEND_MORSE),
    (LongName:'get_dcd';ShortName:#$8B;CmdCode:GET_DCD),
    (LongName:'set_rptr_shift';ShortName:'R';CmdCode:SET_RPTR_SHIFT),
    (LongName:'get_rptr_shift';ShortName:'r';CmdCode:GET_RPTR_SHIFT),
    (LongName:'set_rptr_offs';ShortName:'O';CmdCode:SET_RPTR_OFFS),
    (LongName:'get_rptr_offs';ShortName:'o';CmdCode:GET_RPTR_OFFS),
    (LongName:'set_ctcss_tone';ShortName:'C';CmdCode:SET_CTCSS_TONE),
    (LongName:'get_ctcss_tone';ShortName:'c';CmdCode:GET_CTCSS_TONE),
    (LongName:'set_dcs_code';ShortName:'D';CmdCode:SET_DCS_CODE),
    (LongName:'get_dcs_code';ShortName:'d';CmdCode:GET_DCS_CODE),
    (LongName:'set_ctcss_sql';ShortName:#$90;CmdCode:SET_CTCSS_SQL),
    (LongName:'get_ctcss_sql';ShortName:#$91;CmdCode:GET_CTCSS_SQL),
    (LongName:'set_dcs_sql';ShortName:#$92;CmdCode:SET_DCS_SQL),
    (LongName:'get_dcs_sql';ShortName:#$93;CmdCode:GET_DCS_SQL),
    (LongName:'set_ts';ShortName:'N';CmdCode:SET_TS),
    (LongName:'get_ts';ShortName:'n';CmdCode:GET_TS),
    (LongName:'set_func';ShortName:'U';CmdCode:SET_FUNC),
    (LongName:'get_func';ShortName:'u';CmdCode:GET_FUNC),
    (LongName:'set_level';ShortName:'L';CmdCode:SET_LEVEL),
    (LongName:'get_level';ShortName:'l';CmdCode:GET_LEVEL),
    (LongName:'set_parm';ShortName:'P';CmdCode:SET_PARM),
    (LongName:'get_parm';ShortName:'p';CmdCode:GET_PARM),
    (LongName:'set_bank';ShortName:'B';CmdCode:SET_BANK),
    (LongName:'set_mem';ShortName:'E';CmdCode:SET_MEM),
    (LongName:'get_mem';ShortName:'e';CmdCode:GET_MEM),
    (LongName:'vfo_op';ShortName:'G';CmdCode:VFO_OP),
    (LongName:'scan';ShortName:'g';CmdCode:SCAN),
    (LongName:'set_channel';ShortName:'H';CmdCode:SET_CHANNEL),
    (LongName:'get_channel';ShortName:'h';CmdCode:GET_CHANNEL),
    (LongName:'set_trn';ShortName:'A';CmdCode:SET_TRN),
    (LongName:'get_trn';ShortName:'a';CmdCode:GET_TRN),
    (LongName:'reset';ShortName:'*';CmdCode:RESET),
    (LongName:'set_powerstat';ShortName:#$87;CmdCode:SET_POWERSTAT),
    (LongName:'get_powerstat';ShortName:#$88;CmdCode:GET_POWERSTAT),
    (LongName:'send_dtmf';ShortName:#$89;CmdCode:SEND_DTMF),
    (LongName:'recv_dtmf';ShortName:#$8A;CmdCode:RECV_DTMF),
    (LongName:'get_info';ShortName:'_';CmdCode:GET_INFO),
    (LongName:'get_rig_info';ShortName:#$F5;CmdCode:GET_RIG_INFO),
    (LongName:'get_vfo_info';ShortName:#$F3;CmdCode:GET_VFO_INFO),
    (LongName:'dump_state';ShortName:'';CmdCode:DUMP_STATE),
    (LongName:'dump_caps';ShortName:'1';CmdCode:DUMP_CAPS),
    (LongName:'power2mW';ShortName:'2';CmdCode:POWER2MW),
    (LongName:'mW2power';ShortName:'4';CmdCode:MW2POWER),
    (LongName:'set_clock';ShortName:'';CmdCode:SET_CLOCK),
    (LongName:'get_clock';ShortName:'';CmdCode:GET_CLOCK),
    (LongName:'chk_vfo';ShortName:'';CmdCode:CHK_VFO),
    (LongName:'set_vfo_opt';ShortName:'';CmdCode:SET_VFO_OPT),
    (LongName:'quit';ShortName:'q';CmdCode:QUIT)
  );

  HamlibLevels: THamlibLevels = (
    (LevelName: 'PREAMP'; LevelCode: PREAMP),
    (LevelName: 'ATT'; LevelCode: ATT),
    (LevelName: 'VOX'; LevelCode: LVOX),
    (LevelName: 'AF'; LevelCode: AF),
    (LevelName: 'RF'; LevelCode: LRF),
    (LevelName: 'SQL'; LevelCode: LSQL),
    (LevelName: 'IF'; LevelCode: LIF),
    (LevelName: 'APF'; LevelCode: LAPF),
    (LevelName: 'NR'; LevelCode: LNR),
    (LevelName: 'PBT_IN'; LevelCode: PBT_IN),
    (LevelName: 'PBT_OUT'; LevelCode: PBT_OUT),
    (LevelName: 'CWPITCH'; LevelCode: CWPITCH),
    (LevelName: 'RFPOWER'; LevelCode: RFPOWER),
    (LevelName: 'RFPOWER_METER'; LevelCode: RFPOWER_METER),
    (LevelName: 'RFPOWER_METER_WATTS'; LevelCode: RFPOWER_METER_WATTS),
    (LevelName: 'MICGAIN'; LevelCode: MICGAIN),
    (LevelName: 'KEYSPD'; LevelCode: KEYSPD),
    (LevelName: 'NOTCHF'; LevelCode: NOTCHF),
    (LevelName: 'COMP'; LevelCode: NOTCHF),
    (LevelName: 'AGC'; LevelCode: AGC),
    (LevelName: 'BKINDL'; LevelCode: BKINDL),
    (LevelName: 'BAL'; LevelCode: BAL),
    (LevelName: 'METER'; LevelCode: METER),
    (LevelName: 'VOXGAIN'; LevelCode: VOXGAIN),
    (LevelName: 'ANTIVOX'; LevelCode: ANTIVOX),
    (LevelName: 'SLOPE_LOW'; LevelCode: SLOPE_LOW),
    (LevelName: 'SLOPE_HIGH'; LevelCode: SLOPE_HIGH),
    (LevelName: 'RAWSTR'; LevelCode: RAWSTR),
    (LevelName: 'SWR'; LevelCode: SWR),
    (LevelName: 'ALC'; LevelCode: ALC),
    (LevelName: 'STRENGTH'; LevelCode: STRENGTH)
  );

  HamlibFunctions: THamlibFunctions = (
   (FunctionName: 'FAGC'; FunctionCode: FAGC),
   (FunctionName: 'NB'; FunctionCode: NB),
   (FunctionName: 'COMP'; FunctionCode: COMP),
   (FunctionName: 'VOX'; FunctionCode: FVOX),
   (FunctionName: 'TONE'; FunctionCode: TONE),
   (FunctionName: 'TSQL'; FunctionCode: TSQL),
   (FunctionName: 'SBKIN'; FunctionCode: SBKIN),
   (FunctionName: 'FBKIN'; FunctionCode: FBKIN),
   (FunctionName: 'ANF'; FunctionCode: ANF),
   (FunctionName: 'NR'; FunctionCode: NR),
   (FunctionName: 'AIP'; FunctionCode: AIP),
   (FunctionName: 'APF'; FunctionCode: APF),
   (FunctionName: 'MON'; FunctionCode: MON),
   (FunctionName: 'MN'; FunctionCode: MN),
   (FunctionName: 'RF'; FunctionCode: FRF),
   (FunctionName: 'ARO'; FunctionCode: ARO),
   (FunctionName: 'LOCK'; FunctionCode: LOCK),
   (FunctionName: 'MUTE'; FunctionCode: MUTE),
   (FunctionName: 'VSC'; FunctionCode: VSC),
   (FunctionName: 'REV'; FunctionCode: REV),
   (FunctionName: 'SQL'; FunctionCode: FSQL),
   (FunctionName: 'ABM'; FunctionCode: ABM),
   (FunctionName: 'BC'; FunctionCode: BC),
   (FunctionName: 'MBC'; FunctionCode: MBC),
   (FunctionName: 'RIT'; FunctionCode: RIT),
   (FunctionName: 'AFC'; FunctionCode: AFC),
   (FunctionName: 'SATMODE'; FunctionCode: SATMODE),
   (FunctionName: 'SCOPE'; FunctionCode: SCOPE),
   (FunctionName: 'RESUME'; FunctionCode: RESUME),
   (FunctionName: 'TBURST'; FunctionCode: TBURST),
   (FunctionName: 'TUNER'; FunctionCode: TUNER),
   (FunctionName: 'XIT'; FunctionCode: XIT)
  );

// Global variables
var
  SplitVFOActive: boolean;
  curVFO: TVFONames = VFOA;
  tmpVFO: TVFONames = NONE;
  dbS9: integer;
  ipwr: integer;
  pwr01: real;
  HamlibResult: THamlibErrors;


// Auxiliary procedures & functions

// Send a string via TCP socket
procedure SendNetMsg(Msg:string; MySocket: TLSocket);
begin
  Msg := Msg+stLF;
  MySocket.SendMessage(Msg);
  {$IFDEF DEBUG_NET}
  EKD.Message('Net TX: '+EKD.ShowControlChars(Msg));
  {$ENDIF}
  {$IFDEF DEBUG_PARSE}
  EKD.Message('Reply sent: '+EKD.ShowControlChars(Msg));
  {$ENDIF}
end;

// Parse and execute hamlib commands
procedure ParseNetMsg(RXLine: string; MySocket: TLSocket);

var LineNum, ParamNum, NewFreq: integer;
    sCmd, sParam1, sParam2, sParam3, sTmp: string;
    {$IFDEF THIRDPARAMETER}
    // not required so far
    sParam3: string
    {$ENDIF}
    Cmd: TCmdCodes = LASTCMD;
    Level: TLevels = LASTLEVEL;
    Functn: TFunctions = LASTFUNCTION;
    Success: Boolean = FALSE;

begin
  {$IFDEF DEBUG_PARSE}
  MsgToShow := 'ParseNetMsg entered';
  MsgToShow := 'Line received: '+EKD.ShowControlChars(RXLine);
  {$ENDIF}

  if RXLine='' then exit;

  Cmd := LASTCMD;

  // Remove trailing/ending white spaces and control chars
  RXLine := Trim(RXLine);
  HamlibResult := NOTREQUIRED;

  // Separate command and parameters, if any, and remove remaining spaces
  // and control chars. Accept both LF and SPACE as delimiters
  sCmd := Trim(ExtractDelimited(1,RXLine,[' ',#$0a]));
  sParam1 := Trim(ExtractDelimited(2,RXLine,[' ',#$0a]));
  sParam2 := Trim(ExtractDelimited(3,RXLine,[' ',#$0a]));
  {$IFDEF THIRDPARAMETER}
  // not required so far
  sParam3 := Trim(ExtractDelimited(4,RXLine,[' ',#$0a]));}
  {$ENDIF}
  {$IFDEF DEBUG_PARSE}
  EKD.Message('Cmd: '+sCmd);
  EKD.Message('Param1: '+sParam1);
  EKD.Message('Param2: '+sParam2);
  {$IFDEF NOTDEF}
  // not required so far
  RA1792.Message('Param3: '+sParam3);
  {$ENDIF}
  {$ENDIF}

  // Ignore empty commands
  if sCmd = '' then exit;

  // Check for extended response protocol (NOT SUPPORTED)
  if sCmd[1] in ['+',';','|',','] then begin
    MsgToShow := EXTPROTNOTSUPP;
    SendNetMsg(RPRT+IntToStr(-Ord(RIG_ENIMPL)),MySocket);
    exit;
  end;

  // Parse hamlib long commands
  if sCmd[1]='\' then begin
    // It is a long command
    Delete(sCmd,1,1);
    for ParamNum := 1 to Ord(LASTCMD) do begin
      if sCmd=HamLibCmds[ParamNum].LongName then begin
        Cmd := HamLibCmds[ParamNum].CmdCode;
        break;
      end;
    end;
  end else begin
    // parse hamlib short commands
    for ParamNum := 1 to ord(LASTCMD) do begin
      if sCmd=HamLibCmds[ParamNum].ShortName then begin
         Cmd := HamLibCmds[ParamNum].CmdCode;
         break;
      end;
    end;
  end;

  {$IFDEF DEBUG_PARSE}
  EKD.Message('Received '+HamLibCmds[ParamNum].LongName+' command');
  {$ENDIF}

  // Now we may have a command in Cmd with parameters in sParam1, sParam2 and sParam3

  // Execute command
  // FIXME: At present only VFOMode 0 is handled.
  // Any program out there *wants* VFOMODE 1?
  case Cmd of
    SET_FREQ: begin
      if not In_Command then begin
        // Frequency can have decimal figures (not used here) so remove them if present
        // The separator can be '.' or ',', so handle both
        sParam1 := ExtractDelimited(1,sParam1,['.',',']);
        if VFOMode=0 then begin
          // The only handled so far
          if TryStrToInt(sParam1,NewFreq) then begin
            // round to nearest 10 Hz
            NewFreq := (NewFreq div 10)*10;
            // Check if frequency is in allowable range
            // if outside, set it to lowest/highest allowed value
            if NewFreq < MINRXF then NewFreq := MINRXF;
            if NewFreq > MAXRXF then NewFreq := MAXRXF;

            In_Command := TRUE;
            RXState.Freq_Rx := NewFreq;
            EKD.SetRXFreq;
            In_Command := FALSE;
            HamlibResult := RIG_OK;
            PanelUpdateRequired := TRUE;
          end else HamLibResult := RIG_EINVAL;
        end;
      end else HamLibResult := RIG_BUSBUSY;
    end;
    GET_FREQ: begin
      // TBD correctly for VFOmode 1 if required
      if VFOMode=0 then
        SendNetMsg(IntToStr(RXState.Freq_rx),MySocket);
    end;
    SET_MODE: begin
      if not (In_Command or In_Status) then begin
        In_Command := TRUE;
        // Set mode D
        Success := EKD.SetRemoteMode(Mode_D, DONT_REENABLE);

        if sParam1 <> '' then begin
          // To be completed if needed and possible for all modes supported
          if sParam1='USB' then stmp := '4'
          else if sParam1='LSB' then stmp := '4'
          else if sParam1='CW' then stmp := '1'
          else if sParam1='AM' then stmp := '2'
          else if sParam1='RTTY' then stmp := '8'
          else if sParam1='RTTYR' then stmp := '9'
          else if sParam1='AMS' then stmp := '3'
   //       else if sParam1='ECSSLSB' then stmp := '3'
          else stmp := '';
          if stmp <> '' then begin
            Success := Success and EKD.SendCommandD(Key_MOD, stmp);
            if Success then RXState.Mode_RX := StrToInt(stmp);
          end;
        end;

        // Set bandwidth
        if sParam2 <> '' then begin
          if sParam2 = '150' then  stmp := '1'
          else if sParam2 = '400' then stmp := '2'
          else if sParam2 = '750' then stmp := '3'
          else if sParam2 = '1750' then stmp := '4'
          else if sParam2 = '3100' then stmp := '5'
          else if sParam2 = '6000' then stmp := '6'
          else if sParam2 = '3000' then stmp := '7'
          else if sParam2 = '-3000' then stmp := '8'
          else begin
            stmp := '';
            HamLibResult := RIG_EINVAL;
          end;
          if stmp<>'' then begin
            Success := Success and EKD.SendCommandD(Key_B, stmp);
            if Success then begin
              RXState.Filter := StrToInt(stmp);
              HamlibResult := RIG_OK
            end else
              HamlibResult := RIG_BUSERROR;
          end;
        end;
        // Restore mode C
        EKD.SetRemoteMode(Mode_C, DONT_REENABLE);
        EKD.SetRemoteMode(Mode_C, REENABLE);
        In_Command := FALSE;
        PanelUpdateRequired := TRUE;
      end else HamlibResult := RIG_BUSBUSY;
   end;
    GET_MODE: begin
      sTmp := '';
      case RXState.Mode_RX of
        M_J3: if RXstate.Filter = FILTER_3000U then
               sTmp:='USB'
             else
               sTmp := 'LSB';
        M_A1: sTmp:='CW';
        M_R3: if RXstate.Filter = FILTER_3000U then
                sTmp:='AMS'
              else
                sTmp := 'AMS';
        M_A3: sTmp:='AM';
        M_BR8: sTmp:='DSB';
        M_B8: sTmp:='DSB';
        M_F0,M_F1U: sTmp := 'RTTY';
        M_F1D: sTmp:='RTTYR';
      end;
      // rigctld returns <mode><LF><bandwidth>, so the same is done here.
      sTmp := sTmp+stLF;

      // Values are for my own EKD500
      case RXState.Filter of
        FILTER_150: sTmp := sTmp+'150';
        FILTER_400: sTmp := sTmp+'400';
        FILTER_750: sTmp := sTmp+'750';
        FILTER_1750: sTmp := sTmp+'1750';
        FILTER_3100: sTmp := sTmp+'3100';
        FILTER_6000: sTmp := sTmp+'6000';
        FILTER_3000U: sTmp := sTmp+'+3000';
        FILTER_3000L: sTmp := sTmp+'-3000';
      end;
      SendNetMsg(sTmp,MySocket);
    end;
    SET_VFO: begin
      // Only one VFO available. Report always success.
      HamlibResult := RIG_OK;
    end;
    GET_VFO: begin
      // Only one VFO available, report VFOA.
      SendNetMsg('VFOA',MySocket);
    end;
    SET_RIT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_RIT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_XIT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_XIT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_PTT: begin
      // Does not apply to a receiver but other programs don't know that,
      // so send a OK reply
      HamlibResult := RIG_OK;
    end;
    GET_PTT: begin
      // Does not apply to a receiver but other programs don't know that,
      // so send a 0 reply
      SendNetMSG('0',MySocket);
    end;
    SET_SPLIT_VFO: begin
      // Does not apply, but report always success for WSJTX
      HamlibResult := RIG_OK;
    end;
    GET_SPLIT_VFO: begin
      // No split VFO and VFOA active
      SendNetMSG('0'+stLF+'VFOA',MySocket);
    end;
    SET_SPLIT_FREQ: begin
      // Does not apply
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_SPLIT_FREQ: begin
      // Does not apply
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_SPLIT_MODE: begin
      // Does not apply, report always success for WSJTX
      HamlibResult := RIG_OK;
    end;
    GET_SPLIT_MODE: begin
     // Does not apply
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_ANT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_ANT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SEND_MORSE: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_DCD: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_RPTR_SHIFT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_RPTR_SHIFT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_RPTR_OFFS: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_RPTR_OFFS: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_CTCSS_TONE: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_CTCSS_TONE: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_DCS_CODE: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_DCS_CODE: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_CTCSS_SQL: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_CTCSS_SQL: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_DCS_SQL: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_DCS_SQL: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_TS: begin
      // EKD500 has programmable tune steps in 10 Hz increments
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_TS: begin
      // EKD500 has programmable tune step in 10 Hz increments
      SendNetMsg(IntToStr(RXState.Freq_step), MySocket);
    end;
    SET_FUNC: begin
      // To be completed with all applicable functions
      for ParamNum := 1 to ord(LASTFUNCTION) do begin
        if sParam1=HamlibFunctions[ParamNum].FunctionName then begin
          Functn := HamlibFunctions[ParamNum].FunctionCode;
          break;
        end;
      end;
      // TBD
    end;
    GET_FUNC: begin
       // To be completed with all applicable functions
      for ParamNum := 1 to ord(LASTFUNCTION) do begin
        if sParam1=HamlibFunctions[ParamNum].FunctionName then begin
          Functn := HamlibFunctions[ParamNum].FunctionCode;
          break;
        end;
      end;
      // TBD
    end;
    SET_LEVEL: begin
      // To be completed with all applicable levels
      for ParamNum := 1 to ord(LASTLEVEL) do begin
        if sParam1=HamlibLevels[ParamNum].LevelName then begin
          Level := HamlibLevels[ParamNum].LevelCode;
          break;
        end;
      end;

      case Level of
        AGC: begin
          if not (In_Command or In_Status) then begin
            In_Command := TRUE;
            Success := EKD.SetRemoteMode(Mode_C, TRUE);
            Success := EKD.SetRemoteMode(Mode_C, TRUE);
            case StrToInt(sParam2) of
              0: begin
                Success := Success and EKD.SendCommandC(Key_GC, statusC, DONT_REENABLE);
                Success := Success and EKD.SendCommandC(IntToStr(AVC_MAN), statusC, DONT_REENABLE);
                if Success then RXState.AGC := AVC_MAN;
              end;
              1,2: begin
                Success := Success and EKD.SendCommandC(Key_GC, statusC, DONT_REENABLE);
                Success := Success and EKD.SendCommandC(IntToStr(AVC_FAST), statusC, DONT_REENABLE);
                if Success then RXState.AGC := AVC_FAST;
              end;
              3: begin
                Success := Success and EKD.SendCommandC(Key_GC, statusC, DONT_REENABLE);
                Success := Success and EKD.SendCommandC(IntToStr(AVC_SLOW), statusC, DONT_REENABLE);
                if Success then RXState.AGC := AVC_SLOW;
              end;
            end;
            EKD.SendCommandC(Key_e, statusC, REENABLE);
            PanelUpdateRequired := TRUE;
            In_Command := FALSE;
          end else HamlibResult := RIG_BUSBUSY;
        end;
      end;
    end;
    GET_LEVEL: begin
      // To be completed with all applicable levels
      for ParamNum := 1 to ord(LASTLEVEL) do begin
        if sParam1=HamlibLevels[ParamNum].LevelName then begin
          Level := HamlibLevels[ParamNum].LevelCode;
          break;
        end;
      end;

      case Level of
        AGC: begin
          case RXstate.AGC of
            AVC_FAST, AVC_MAN_FAST: SendNetMsg('2',MySocket);
            AVC_SLOW, AVC_MAN_SLOW: SendNetMsg('3',MySocket);
            AVC_MAN: SendNetMsg('0',MySocket);
          end;
        end;
        STRENGTH: begin
          // Level returned by the GC command is 0-63 in 2dBuV units
          // dBm is 0-120dBuV, then S9=100uV=40dBuV, 0 is 1uV = ~S3
          dBS9 := dBm-40;
          SendNetMsg(IntToStr(dBS9),MySocket);
        end;
      end;
    end;
    SET_PARM: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    GET_PARM: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    SET_BANK: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_MEM: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_MEM: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    VFO_OP: begin
      // TBD for ops supported, if any, and if *needed* by some program
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    SCAN: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    SET_CHANNEL: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    GET_CHANNEL: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    SET_TRN: begin
      // Not available (?)
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_TRN: begin
      // Not available (?)
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    RESET: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_POWERSTAT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_POWERSTAT: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SEND_DTMF: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    RECV_DTMF: begin
      // Not available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_INFO: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    GET_RIG_INFO: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    GET_VFO_INFO: begin
      // TBD if some program *needs* it
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    DUMP_STATE: begin
      // TBD in the correct way - See UState.pas
      LineNum := -1;
      repeat
        LineNum := Linenum + 1;
        SendNetMsg(EKD500_State[LineNum],MySocket);
      until EKD500_State[LineNum]='done';
    end;
    DUMP_CAPS: begin
      // TBD if required
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    POWER2MW: begin
      // Does not apply
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    MW2POWER: begin
      // Does not apply
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    SET_CLOCK: begin
      // No RTC available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    GET_CLOCK: begin
      // No RTC available
      MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamlibResult := RIG_ENAVAIL;
    end;
    CHK_VFO: begin
      // Required by WSJTX and other programs
      // Only Mode 0 at present is handled, so VFOMode is always 0
      SendNetMsg(IntToStr(VFOMode),MySocket);
    end;
    SET_VFO_OPT: begin
      // TDB if any and if it is *needed* by some program
      MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamlibResult := RIG_ENIMPL;
    end;
    QUIT: begin
      SendNetMsg(RPRT_OK,MySocket);
      MySocket.Disconnect(FALSE);
      PanelUpdateRequired := TRUE;
    end;
    otherwise begin
      // Command not recognized
      MsgToShow := Format(REMCMDNOTREC,[sCmd]);
      HamlibResult := RIG_EINVAL;
    end;
  end;
  // If required return response
  if HamlibResult <> NOTREQUIRED then
    SendNetMsg(RPRT+IntToStr(-Ord(HamlibResult)),MySocket);
end;

end.

