{
    This file is part of EKD500Control

    RFT EKD500 Control program

    Copyright (C) 2014-2025 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/.
}

unit EKD500;

{$mode objfpc}{$H+}

{$INCLUDE defines.inc}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, StdCtrls,
  Menus, ExtCtrls, Buttons, serial, types, LCLType, ActnList, Spin, math, mouse,
  dateutils, user, UAbout, UGNU, UMAN, urxaddr, umanager, unrx, uscan, uTCP,
  JButton, JLabel, lNetComponents, FileInfo, LCLTranslator, DefaultTranslator,
  ComboEx, lNet, uparse, XMLConf;

type

  { TEKD }

  TEKD = class(TForm)
    B0: TJButton;
    B1: TJButton;
    B2: TJButton;
    B3: TJButton;
    B4: TJButton;
    B5: TJButton;
    B6: TJButton;
    B7: TJButton;
    B8: TJButton;
    B9: TJButton;
    BARROW: TJButton;
    BB: TJButton;
    BCALL: TJButton;
    BCALL98: TJButton;
    BCALL99: TJButton;
    BDELTAF: TJButton;
    BDOT: TJButton;
    BE: TJButton;
    BEXT: TJButton;
    BEXTFCT: TJButton;
    BF: TJButton;
    BGC: TJButton;
    BMOD: TJButton;
    BSCAN: TJButton;
    BSCANFCT: TJButton;
    BSEL: TJButton;
    BSTO: TJButton;
    CBAGC: TComboBoxEx;
    CBBW: TComboBoxEx;
    CBMODE: TComboBoxEx;
    CBREC: TComboBoxEx;
    CBSEL: TComboBoxEx;
    GRXFreq: TGroupBox;
    LBA: TJLabel;
    LBBW: TJLabel;
    LBDBM: TJLabel;
    LBDBM1: TJLabel;
    LBDBM2: TJLabel;
    LBDBM3: TJLabel;
    LBNETW: TLabel;
    LBEXT: TLabel;
    LBF: TJLabel;
    LBM: TJLabel;
    LBMOD: TLabel;
    LBB: TLabel;
    LBDBuV: TLabel;
    LBKHZ: TLabel;
    LBR: TJLabel;
    LBS: TJLabel;
    LBSIG1: TJLabel;
    LBSIG2: TJLabel;
    LBSIG3: TJLabel;
    EKD500server: TLTCPComponent;
    MADDRP: TMenuItem;
    MSHOWHAMLIB: TMenuItem;
    Separator2: TMenuItem;
    MMAGALLF: TMenuItem;
    Separator1: TMenuItem;
    MMAG175: TMenuItem;
    MMAG100: TMenuItem;
    MMAG110: TMenuItem;
    MMAG125: TMenuItem;
    MMAG150: TMenuItem;
    MMAG200: TMenuItem;
    MBSCANW: TMenuItem;
    MenuItem8: TMenuItem;
    N3: TMenuItem;
    N2: TMenuItem;
    N1: TMenuItem;
    MNUMRX: TMenuItem;
    MenuItem2: TMenuItem;
    MenuItem3: TMenuItem;
    MRSTATE: TMenuItem;
    MMANCH: TMenuItem;
    MMANST: TMenuItem;
    MSCHANN: TMenuItem;
    MRXADD: TMenuItem;
    M200B: TMenuItem;
    M600B: TMenuItem;
    M1200B: TMenuItem;
    MSG: TMemo;
    MenuItem11: TMenuItem;
    MChannels: TMenuItem;
    MSHOWC: TMenuItem;
    MRed: TMenuItem;
    MGreen: TMenuItem;
    MYellow: TMenuItem;
    MenuItem6: TMenuItem;
    MenuItem7: TMenuItem;
    MLEDC: TMenuItem;
    MenuItem9: TMenuItem;
    MENALL: TMenuItem;
    MGNULIC: TMenuItem;
    MABOUT: TMenuItem;
    MMAN: TMenuItem;
    MHelp: TMenuItem;
    MLSTATE: TMenuItem;
    MSSTATE: TMenuItem;
    MenuItem5: TMenuItem;
    MSmeter: TMenuItem;
    MenuItem1: TMenuItem;
    MCustom: TMenuItem;
    OpenDialog1: TOpenDialog;
    RXFreq100H: TLabel;
    RXFreq100k: TLabel;
    RXFreq10H: TLabel;
    RXFreq10k: TLabel;
    RXFreq10M: TLabel;
    RXFreq1k: TLabel;
    RXFreq1M: TLabel;
    RXFreqB: TLabel;
    RXFreqDOT: TLabel;
    RXFreqE: TLabel;
    RXFREQL2: TLabel;
    RXFREQDOT1: TLabel;
    RXFreqMod: TLabel;
    SaveDialog1: TSaveDialog;
    SHEXT: TShape;
    SMGroupBox: TGroupBox;
    SMLabel: TLabel;
    SMShape1: TShape;
    SMShape10: TShape;
    SMShape11: TShape;
    SMShape12: TShape;
    SMShape13: TShape;
    SMShape2: TShape;
    SMShape3: TShape;
    SMShape4: TShape;
    SMShape5: TShape;
    SMShape6: TShape;
    SMShape7: TShape;
    SMShape8: TShape;
    SMShape9: TShape;
    SPFREQ: TSpinEdit;
    SPDELTAF: TSpinEdit;
    LBP: TJLabel;
    TNET: TTimer;
    TSMDisp: TTimer;
    TSmeter: TTimer;
    MainMenu1: TMainMenu;
    MFile: TMenuItem;
    M300b: TMenuItem;
    M2400B: TMenuItem;
    MExit: TMenuItem;
    MOptions: TMenuItem;
    MPort: TMenuItem;
    MSpeed: TMenuItem;
    Mttys0: TMenuItem;
    Mttys1: TMenuItem;
    XMLConfig1: TXMLConfig;
    procedure BARROWClick(Sender: TObject);
    procedure BCALL98Click(Sender: TObject);
    procedure BDOTClick(Sender: TObject);
    procedure BFClick(Sender: TObject);
    procedure BCALL99Click(Sender: TObject);
    procedure BCALLClick(Sender: TObject);
    procedure BSCANClick(Sender: TObject);
    procedure BSELClick(Sender: TObject);
    procedure BEClick(Sender: TObject);
    procedure BDELTAFClick(Sender: TObject);
    procedure BNumClick(Sender: TObject);
    procedure BEXTClick(Sender: TObject);
    procedure BMODClick(Sender: TObject);
    procedure BBClick(Sender: TObject);
    procedure BGCClick(Sender: TObject);
    procedure BSTOClick(Sender: TObject);
    procedure BEXTFCTClick(Sender: TObject);
    procedure BSCANFCTClick(Sender: TObject);
    procedure CBAGCChange(Sender: TObject);
    procedure CBBWChange(Sender: TObject);
    procedure CBMODEChange(Sender: TObject);
    procedure CBRECChange(Sender: TObject);
    procedure CBSELChange(Sender: TObject);
    procedure EKD500serverAccept(aSocket: TLSocket);
    procedure EKD500serverDisconnect(aSocket: TLSocket);
    procedure EKD500serverError(const amsg: string; aSocket: TLSocket);
    procedure EKD500serverReceive(aSocket: TLSocket);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormKeyPress(Sender: TObject; var Key: char);
    procedure FormResize(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure M1200BClick(Sender: TObject);
    procedure M200BClick(Sender: TObject);
    procedure M600BClick(Sender: TObject);
    procedure MABOUTClick(Sender: TObject);
    procedure MADDRPClick(Sender: TObject);
    procedure MENALLClick(Sender: TObject);
    procedure MFontMagnClick(Sender: TObject);
    procedure MBSCANWClick(Sender: TObject);
    procedure MMAGALLFClick(Sender: TObject);
    procedure MMANCHClick(Sender: TObject);
    procedure MMANSTClick(Sender: TObject);
    procedure MNUMRXClick(Sender: TObject);
    procedure MRSTATEClick(Sender: TObject);
    procedure MRXADDClick(Sender: TObject);
    procedure MSGDblClick(Sender: TObject);
    procedure MGreenClick(Sender: TObject);
    procedure MRedClick(Sender: TObject);
    procedure MGNULICClick(Sender: TObject);
    procedure MLSTATEClick(Sender: TObject);
    procedure MMANClick(Sender: TObject);
    procedure MExitClick(Sender: TObject);
    procedure MSHOWCClick(Sender: TObject);
    procedure MSHOWHAMLIBClick(Sender: TObject);
    procedure MSSTATEClick(Sender: TObject);
    procedure MSmeterClick(Sender: TObject);
    procedure MYellowClick(Sender: TObject);
    procedure RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer);
    procedure RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
              WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure SPDELTAFChange(Sender: TObject);
    procedure SPFREQChange(Sender: TObject);
    procedure TNETTimer(Sender: TObject);
     procedure TSMDispTimer(Sender: TObject);
    procedure TSmeterTimer(Sender: TObject);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState;
              WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure M2400BClick(Sender: TObject);
    procedure M300BClick(Sender: TObject);
    procedure MCustomClick(Sender: TObject);
    procedure Mttys0Click(Sender: TObject);
    procedure Mttys1Click(Sender: TObject);
    procedure KeyboardHandler(keyp: string);
    procedure DispRXf;
    procedure DispRXm;
    procedure UpdateControls;
    function SendCommandD(cmd, par: string): boolean;
    function SendCommandC(cmd: string; var response: string; Reenable_Status: boolean): boolean;
    function OpenRemote: boolean;
    function CloseRemote: boolean;
    function SetRXFreq: boolean;
    procedure GetStatus;
    procedure LoadChannel(Sender: TObject);
    procedure RestoreState;
    procedure SaveConfig;
    procedure ReadConfig;
    procedure ReadConfig_old;
    procedure DisableAllControls;
    procedure EnableAllControls;
    procedure SetActiveButtons(Sender: TObject);
    procedure CloseProgram;
    procedure LoadChannelsDir;
    procedure UpdateDisplay;
    function  SetRemoteMode(Mode: string; Flush: boolean): boolean;
    procedure ParseStatus(Status: string; StatusType: boolean);
    procedure DisableAllButtons;
    procedure DisableKeypad;
    procedure EnableKeypad;
    procedure EnableCmdKeys;
    procedure EnableCombosAndEdits;
    procedure SpinEditUpdateOnly(Sender: TObject; value: integer);
    procedure CalibrateDelays;
    function AdjustDecimalSeparator(s: string): string;
    procedure SetSMeter(value: integer);
    procedure SetSMeterColor;
    procedure SetFreqColor;
    function CalcSValue(dbmv: integer): string;
    function GetRxstring: string;
    procedure PutRXString(s: string);
    procedure FontMagnify;
    function ShowControlChars(cmd: string): string;
    procedure SetDefaultConfig;
    procedure DoMessages;
  private
    { private declarations }
    procedure Message(ms: string);
  public
    { public declarations }
    property MsgToShow: string write Message;
  end;


Const
    //
    // EKD500 command values
    //
    Mode_A:      string = 'A';   // Answer mode 1 (command echo + level after e)
    Mode_B:      string = 'B';   // Answer mode 2 (ACK reply + frequency & level)
    Mode_C:      string = 'C';   // Answer mode 3 (ACK reply + frequency & mode)
    Mode_D:      string = 'D';   // Answer mode 4 (no reply, only errors)
    Tune_Up:     string = '+';   // Tune one step up
    Tune_Dn:     string = '-';   // Tune one step down
    Key_F:       string = 'f';   // frequency in kHz to 10 Hz (xxxxx.xx)
    Key_DF:      string = 'd';   // frequency step in kHz (any value, min 10 Hz)
    Key_MOD:     string = 'm';   // RX Mode (1..9, see table on RX front panel)
    Key_B:       string = 'b';   // IF bandwidth (1..9, see table on RX front panel)
    Key_GC:      string = 'g';   // AGC mode (1..9, see table on RX front panel)
    Key_SEL:     string = 'v';   // Preselector on/off (0..1, 0=off, 1=on)
    Key_CALL:    string = 'c';   // Memory recall (0..99)
    Key_STO:     string = 's';   // Memory store  (0..99)
    Key_CALL98:  string = '"';   // Memory 98 fast recall
    Key_CALL99:  string = '!';   // Memory 99 fast recall
    Key_SCAN:    string = 'l';   // Scan memories or frequency range
    Key_SCANFCT: string = 'a';   // Configure scan parameters
    Key_E:       string = 'e';   // Execute key
    Key_DOT:     string = '.';   // Decimal separator
    Key_ARROW:   string = ';';   // Configuration done (manual at page 57 says 3BH,
                                 // which is correct, and j, NO!, 3BH = ;)

    // Filenames
    XMLConfigFileName: string = 'Config.xml';
    ConfigFileName: string = 'Config.dat';
    BSCANSDIR = 'BandScans';

    // Mode values
    M_A1  = 1;
    M_A3  = 2;
    M_R3  = 3;
    M_J3  = 4;
    M_Br8 = 5;
    M_B8  = 6;
    M_F0  = 7;
    M_F1U = 8;
    M_F1D = 9;

    // Filter values
    FILTER_150   = 1;
    FILTER_400   = 2;
    FILTER_750   = 3;
    FILTER_1750  = 4;
    FILTER_3100  = 5;
    FILTER_6000  = 6;
    FILTER_3000U = 7;
    FILTER_3000L = 8;

    // AGC values
    AVC_FAST = 1;
    AVC_SLOW = 2;
    AVC_MAN_FAST = 3;
    AVC_MAN_SLOW = 4;
    AVC_MAN = 5;

    // Preselector (SEL) values
    SEL_WIDE = 0;
    SEL_NARROW = 1;

    // frequency coverage
    MINRXF = 0;
    MAXRXF = 29999990;

    // Status subtypes
    TYPE_E = 1;
    TYPE_D = 2;
    TYPE_GC = 3;
    TYPE_SE = 4;
    TYPE_SCAN = 5;
    TYPE_C = 6;
    TYPE_S = 7;
    TYPE_A = 8;

    // Button tags
    T_NONE = -1;
    T_B0 = 0;
    T_B1 = 1;
    T_B2 = 2;
    T_B3 = 3;
    T_B4 = 4;
    T_B5 = 5;
    T_B6 = 6;
    T_B7 = 7;
    T_B8 = 8;
    T_B9 = 9;
    T_BDOT = 10;
    T_BARROW = 11;
    T_BEXT = 20;
    T_BMOD = 21;
    T_BB = 22;
    T_BGC = 23;
    T_BSEL = 24;
    T_BEXTFCT = 25;
    T_BSCANFCT = 26;
    T_BSCAN = 27;
    T_BDELTAF = 28;
    T_BF = 29;
    T_BCALL99 = 30;
    T_BCALL98 = 31;
    T_BCALL = 32;
    T_BSTO = 33;
    T_BE = 34;

    // special characters
    ETB: char = #$17;
    ACK: char = #$06;
    NAK: char = #$15;
    stLF: string = #$0a;

    // TRUE/FALSE Constants
    STATUSTYPE_C: boolean = TRUE;
    STATUSTYPE_B: boolean = FALSE;
    FLUSH_INPUT:  boolean = TRUE;
    NOFLUSH:      boolean = FALSE;
    ENABLED:      boolean = TRUE;
    DISABLED:     boolean = FALSE;
    NOERROR:      boolean = TRUE;
    ERROR:        boolean = FALSE;
    DONT_REENABLE:boolean = FALSE;
    REENABLE:     boolean = TRUE;

    // Timeout values
    READTIMEOUT = 1000 * OneMillisecond;

    // Object resize array max components number
    MAXCOMPONENTS = 250;


Type
  TRXState = record
    Freq_rx: longint;
    Freq_step: integer;
    Mode_RX: integer;
    Filter: integer;
    AGC: integer;
    Preselector: integer;
    Scan: boolean;
  end;

var
  EKD: TEKD;
  ProgVersion: TVersionQuad;
  Major, Minor, Revision, Build: integer;
  ShortVersion, LongVersion: string;
  RXState, OldRXState: TRXState;
  SerPort: TSerialHandle = 0;
  SerPortName: string = '/dev/ttyS0';
  SerPortSpeed: integer = 2400;
  SleepTime: integer;
  TimeOneCharMs: integer;
  CurRXNumber: integer = 1;
  CurRXAddress: integer = 1;
  NumberOfReceivers: integer = 1;
  ReceiversAddresses: array[1..99] of integer;
  s,t, dBm: integer;
  statusG: string = '';
  statusC: string = '';
  RXFBuf, OldRXFreq: string;
  sMode_RX: string;
  sMode_TX: string;
  Enable_status: boolean = FALSE;
  In_Wheel: boolean = FALSE;
  In_Status: boolean = FALSE;
  In_RXFreqMouseUp: boolean = FALSE;
  In_TXFreqMouseUp: boolean = FALSE;
  In_Command: boolean = FALSE;
  ENotRequired: boolean = FALSE;
  Update_Only: boolean = FALSE;
  tmp: longint;
  HomeDir,StateDir,BandsDir,ChannelsDir,DocDir,BandScansDir,GPLv3Dir: string;
  timeout: TTime;
  LEDColorOFF: TColor = clGreen;
  LEDColorON: TColor = ClLime;
  sLEDColor: string = 'green';
  HaveConfig: boolean = FALSE;
  n,m: integer;
  BandFileName: string;
  RemoteMode: string;
  CmdTag: integer = -1;
  PrevTag: integer = -1;
  DigitCoord: array[1..8, 1..2] of integer;
  SMShapes: array[1..13] of TShape;
  Errmsg: string;
  FontMagn: integer = 100;

  {$IFDEF AUTOSCALE}
  // Variables used by autoscale code
  OH, OW: integer;
  Xs: array[0..MAXCOMPONENTS] of integer;
  Ys: array[0..MAXCOMPONENTS] of integer;
  Ws: array[0..MAXCOMPONENTS] of integer;
  Hs: array[0..MAXCOMPONENTS] of integer;
  Wt: array[0..MAXCOMPONENTS] of Integer;
  Ht: Integer;
  Wc: array[0..MAXCOMPONENTS] of Integer;
  Hc: Integer;
  Wn: array[0..MAXCOMPONENTS] of Integer;
  Hn: Integer;
  Hf: array[0..MAXCOMPONENTS] of Integer;
  {$ENDIF}
  PrevHeight, PrevWidth: integer;
  StartHeight, StartWidth: integer;
  StartX, StartY: integer;
  PPI_X, PPI_Y: integer;

  // Variables for LNET rigctl server
  RXLine: string;
  NumTCPConnections: integer = 0;
  RemFreqChange: boolean = FALSE;
  PanelUpdateRequired: boolean = FALSE;
  MsgToShow: string  = '';


// Declared here for i18n
resourcestring
  // Messages
  CANTOPENCHANFILE = 'Can''t open channel file.';
  CANTOPENSTATEFILE = 'Can''t open state file %s.';
  CANTWRITESTATEFILE = 'Can''t write state file %s.';
  CANTWRITECONF = 'Can''t write configuration file.';
  CHANMGR = 'Channel Manager';
  CHECKINST = 'Check your installation.';
  CLPROGRAM = 'Close program';
  COMMANDFAILED = 'Command failed.';
  CONFIGNOTFOUND = 'Config file not found, loading and saving'+LineEnding+
                   'default configuration.';
  COPYRIGHTST = ' (C) 2012-2025 G. Perotti, I1EPJ';
  DISABLINGSTATUS = 'Disabling status reading.';
  DOCFILESNOTF = 'Manual and/or license files not found.';
  DONE = 'Done!';
  ENTERAGC = 'Enter AGC mode (1..5).';
  ENTERBANDWIDTH = 'Enter bandwidth (1..9).';
  ENTERCHANNEL = 'Enter channel# (0..99)';
  ENTERF = 'Enter frequency in kHz.';
  ENTERFSTEP = 'Enter frequency step in kHz.';
  ENTERMODE = 'Enter mode (1..9).';
  ENTERPRESTYPE = 'Enter preselector type (0..1).';
  ENTERSCANPARAMS = 'Enter scan parameters (see manual).';
  ERRORINIT = 'Failed to initialize communications.'+LineEnding+
              'Retry?';
  ERRORNET = 'Net error occurred: %s';
  ERROROPENCOM = 'Unable to open serial port "%s.".'+LineEnding+
                 'Check name and user privileges.'+LineEnding+
                 'Retry?';
  ERRORMSG = 'Error';
  ERRORREADINGCONF = 'Error reading configuration';
  GETPROGVFAIL = 'GetProgramVersion failed, please enable'+LineEnding+
                 'version informations in Project Options.';
  LISTENFAILED = 'EKD500server socket: listen() failed';
  NOREMOTECMD = 'No remote command, showing program configuration.';
  NORESPONSE = 'No response.';
  NORXAT = 'No RX #%d at addr %d.';
  NOSQUAREPIXELS = 'This screen does not have square pixels:'+LineEnding+
                   'PPI_X=%d PPI_Y=%d.'+LineEnding+
                   'Expect strange behaviour in scaling routines.';
  NOVERSINFO = 'No version info.';
  OLDUPDATING = 'Old format config file found, read and updated to the new XML format.';
  OLDSTYLECONF = 'Old style config file found, updating and saving.';
  PRGWILLHALT = 'The program will be halted.';
  READY = 'Ready!';
  RECEIVED = '%s received, cmd=%s response=%s.';
  REALLYEXIT = 'Really exit?';
  RECVD = '%s received.';
  RESTORINGSTATE = 'Restoring state...';
  RXCANTTUNE = 'RX can''t tune to that frequency.';
  SENDCOMMANDCTIMEOUT = 'Timeout in SendCommandC, response=%s.';
  SPEEDTOOSLOW = 'Serial speed too slow!'+LineEnding+'Smeter reading disabled.';
  SWITCHINGTORX = 'Switching to receiver #%d...';
  STARTINGPROGRAM = 'Starting EKD500Control...';
  STATEMGR = 'State Manager';
  UNSUPPORTEDCONFIG = 'WARNING: unsupported config file version %s'+
                       LineEnding+
                       'Triyng to read it anyway.';
implementation

{$R *.lfm}

{ TEKD }

//
// Auxiliary procedures & functions
//

// Make control chars visible in command strings
function TEKD.ShowControlChars(cmd: string): string;

var chp: integer;
    stmp: string = '';
    chtmp: string;
begin
  // Display control chars in command strings (see MSHOWCClick)
  for chp := 1 to length(cmd) do begin
    case cmd[chp] of
      {$IFDEF WIN32}
      // No control chars unicode symbols in Win32 (XP)
      // Show only CR as paragraph symbol
      #$0D: chtmp := '¶';
      otherwise chtmp := cmd[chp];
      {$ELSE}
      #$01: chtmp := '␁';
      #$02: chtmp := '␂';
      #$03: chtmp := '␃';
      #$04: chtmp := '␄';
      #$06: chtmp := '␆';
      #$07: chtmp := '␇';
      #$0A: chtmp := '␊';
      #$0D: chtmp := '␍';
      #$10: chtmp := '␐';
      #$15: chtmp := '␕';
      #$18: chtmp := '␘';
      otherwise chtmp := cmd[chp];
      {$ENDIF}
    end;
    stmp := stmp+chtmp;
  end;
  ShowControlChars := stmp;
end;

procedure TEKD.FontMagnify;

var c: TControl;
    i, NewHeight1, NewHeight2, NewHeight: integer;

begin
  {$IFNDEF DISABLE_FONTSCALE}
  for i := 0 to ComponentCount-1 do begin
    if (Components[i] is TLabel) or
       (Components[i] is TStaticText) or
       (Components[i] is TMemo) or
       (Components[i] is TJButton) or
       (Components[i] is TJLabel) or
       (Components[i] is TComboBoxEx) or
       (Components[i] is TSpinEdit) then begin
       {$IFDEF DEBUG_AUTOSCALE}
       Message('scaling font');
       {$ENDIF}
       c := TControl(Components[i]);

       // Do not magnify font of labels with tag <> 0
       if not MMAGALLF.Checked and (c is Tlabel) and (c.tag <> 0) then begin
         NewHeight1 := Hf[i] * ClientWidth div OW;
         NewHeight2 := Hf[i] * ClientHeight div OH;
       end else begin
          NewHeight1 := Hf[i] * ClientWidth * FontMagn div (OW * 100);
          NewHeight2 := Hf[i] * ClientHeight * FontMagn div (OH * 100);
       end;
       NewHeight := min(abs(NewHeight1), abs(NewHeight2));
       if (Components[i] is TJButton) then begin
         {$IFDEF DEBUG_FONTSCALE}
         Message('OW='+inttostr(OW)+' OH='+inttostr(OH));
         Message('ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight));
         Message('Width='+IntToSTr(Width)+', Height='+IntToStr(Height));
         Message('NewHeight='+IntToStr(NewHeight));
         {$ENDIF}
         (Components[i] as TJButton).LCaption.Font.Height := NewHeight;
       end else begin
         c.Font.Height := NewHeight;
       end;
    end;
  end;
  {$ENDIF}
end;

// Get RX string - Makes a string from RX frequency display
// (made by 11 single-character TLabels)
function TEKD.GetRXString: string;

begin
  GetRXString := Trim(RXFreqMod.Caption+RXFReqB.Caption+' '+RXFreqE.Caption+' '+
                 RXFreq10M.Caption+RXFreq1M.Caption+RXFreq100k.Caption+RXFreq10k.Caption+
                 RXFreq1k.Caption+RXFreqDOT.Caption+RXFreq100H.Caption+
                 RXFreq10H.Caption);
end;

//Put RX string - Put a string into RX frequency display right-aligned
procedure TEKD.PutRXString(s: string);

var st: string;

begin
  st := s;
  if Length(st) < 13 then st := RightStr('             '+s,13);
  RXFreqMod.Caption := st[1];
  RXFreqB.Caption := st[2];
  RXFreqDOT1.Caption := st[3];
  RXFreqE.Caption := st[4];
  RXFreqL2.Caption := st[5];
  RXFreq10M.Caption := st[6];
  RXFreq1M.Caption := st[7];
  RXFreq100k.Caption := st[8];
  RXFreq10k.Caption := st[9];
  RXFreq1k.Caption := st[10];
  RXFreqDOT.Caption := st[11];
  RXFreq100H.Caption := st[12];
  RXFreq10H.Caption := st[13];
end;

// S-meter visualization. Make visible the signal meter LEDS (=shapes)
// until value, make invisible LEDS from value+1 to 20.
procedure TEKD.SetSMeter(value: integer);

var i: integer;

begin
  for i := 1 to value do SMShapes[i].Visible := true;
  for i := value+1 to 13 do SMShapes[i].Visible := false;
end;

// Set Smeter color
procedure TEKD.SetSMeterColor;

var i: integer;

begin
  for i:=1 to 13 do begin
    SMShapes[i].Brush.Color := LEDColorON;
    SMShapes[i].Pen.Color := LEDColorON;
  end;
end;

procedure TEKD.SetFreqColor;

begin
        RXFreqMod.Font.Color := LedColorOn;
      RXFreqB.Font.Color := LedColorOn;
      RXFreqE.Font.Color := LedColorOn;
      RXFreq10M.Font.Color := LedColorOn;
      RXFreq1M.Font.Color := LedColorOn;
      RXFreq100k.Font.Color := LedColorOn;
      RXFreq10k.Font.Color := LedColorOn;
      RXFreq1k.Font.Color := LedColorOn;
      RXFreqDot.Font.Color := LedColorOn;
      RXFreqDot1.Font.Color := LedColorOn;
      RXFreqL2.Font.Color := LedColorOn;
      RXFreq100H.Font.Color := LedColorOn;
      RXFreq10H.Font.Color := LedColorOn;
      SHext.Brush.Color := LedColorOn;
end;

function TEKD.CalcSValue(dbmv: integer): string;

var itmp: integer;

begin
  CalcSvalue := 'S';
  itmp := dbmv;
  if itmp >= 40 then begin
    CalcSValue := CalcSValue+'9';
    itmp := itmp-40;
    if itmp > 0 then CalcSValue := CalcSVAlue+'+'+IntTostr(itmp);
  end else begin
    itmp := (itmp div 6)+1;
    CalcSValue := CalcSvalue+IntTostr(itmp);
  end;
end;


// Adjust decimal separator in real number string
// Latest compilers do not accept anymore both separators (. and ,)
function TEKD.AdjustDecimalSeparator(s: string): string;

var i: integer;

begin
  for i := 1 to Length(s) do
    if s[i]='.' then s[i] := DefaultFormatSettings.DecimalSeparator;
    AdjustDecimalSeparator := s;
end;

// Calibrate delays to serial speed
procedure TEKD.CalibrateDelays;
begin
  // Duration of one char in milliseconds
  // There are 11 bit/char (1 start, 7 data, 1 parity, 2 stop)
  // TimeOneCharMs(200 baud) = 55 ms ... TimeOneCharMs(2400 baud) = 4,58 ms
  TimeOneCharMs := 11000 div SerPortSpeed;

  // Delay time (15 ms more than the 200-baud character send time)
  SleepTime := 70-(TimeOneCharMs);
end;

// Process message queue
procedure TEKD.DoMessages;
begin
  Application.ProcessMessages;
end;

// Open communications
function TEKD.OpenRemote: boolean;

var buf: char = char(0);
    nc: integer;
    s: string = '';

begin
  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}
  // Check for EKD500 responding. Set Mode A and wait for ACK.
  SetRemoteMode(Mode_A, FLUSH_INPUT);
  Result := SetRemoteMode(Mode_A, NOFLUSH);
  if Result then begin
    buf := chr(0);
    repeat
      timeout := time + READTIMEOUT;
      repeat
        nc := SerRead(SerPort, buf, 1);
        {$IFDEF LINUX}
        sleep(1);
        {$ENDIF}
        {$IFDEF WINDOWS}
        // under windows minimum sleep is 15ms
        sleep(0);
        {$ENDIF}
      until (nc > 0) or (time > timeout);
    until (buf = ACK) or (time > timeout);

    // no ACK or timeout
    if (buf <> ACK) or (time > timeout) then begin
      Message(NORESPONSE);
      {$IFDEF TEST}
      if time>timeout then Message('(timeout)');
      if buf <> ACK then Message('Received: '+buf);
      {$ENDIF}
      exit(ERROR);
    end;

    // Switch to mode C
    if SetRemoteMode(Mode_C, FLUSH_INPUT) then begin
      SendCommandC(Key_E, s, DONT_REENABLE);
      SendCommandC(Key_E, s, DONT_REENABLE);
      SendCommandC(Key_E, s, REENABLE);
      GetStatus;
      UpdateDisplay;
    end;
    {$IFDEF TEST}
    Message('C:'+StatusC);
    {$ENDIF}
    OpenRemote := NOERROR;
  end else begin
    OpenRemote := ERROR;
  end;
end;

// Close communications
function TEKD.CloseRemote: boolean;
begin
  // ANYTHING TO DO?
  CloseRemote := TRUE;
end;

// Set EKD500 remote mode
function TEKD.SetRemoteMode(Mode: string; Flush: boolean): boolean;

var nc: integer;
    s: string;
    buf: char = #0;

begin
  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}
  s := Mode+IntToHex(CurRXAddress, 2);
  {$IFDEF TEST}
  Message('Set Mode: '+s);
  {$ENDIF}
  sleep(TimeOneCharMs+5);
  nc := SerWrite(SerPort,s[1], 1);
  sleep(TimeOneCharMs+5);
  nc := nc + SerWrite(SerPort,s[2], 1);
  sleep(TimeOneCharMs+5);
  nc := nc + SerWrite(SerPort,s[3], 1);
  sleep(TimeOneCharMs+5);
  if nc <> 3 then begin
    {$IFDEF TEST}
    Message('Set mode ' + Mode + ' failed');
    {$ENDIF}
    exit(ERROR);
  end;

  if Flush then begin
    sleep(30*TimeOneCharMs);
    statusC := '';
    while SerRead(SerPort, buf, 1) > 0 do begin
      {$IFDEF TEST}
      if buf = NAK then Message('SetMode: NAK received');
      {$ENDIF}
      statusC := buf + statusC;
    end;
  end;
  SetRemoteMode := NOERROR;
  RemoteMode := Mode[1];
end;

// Mode C SendCommand (send a single character and returns response)
function TEKD.SendCommandC(cmd: string; var response: string; Reenable_Status: boolean): boolean;

var nc: integer;
    buf: char = #0;
    stmp: string;

begin
  Enable_Status := DISABLED;
  stmp := cmd;

  if MSHOWC.Checked then Message('CmdC: '+cmd);

  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}

//  sleep(SleepTime);
  nc := SerWrite(SerPort, cmd[1], 1);
  {$IFDEF TEST}
  if nc <> 1 then begin
    Message('SerWrite returned ' + IntToStr(nc));
    exit(ERROR);
  end;
  {$ENDIF}
  {$IFDEF TESTHANDSHAKE}
    if SerGetDSR(SerPort) then Message('DSR: TRUE') else Message('DSR: FALSE');
    if SerGetCTS(SerPort) then Message('CTS: TRUE') else Message('CTS: FALSE');
  {$ENDIF}

  stmp := '';
  repeat
    timeout := time + READTIMEOUT;
    repeat
      nc := SerRead(SerPort, buf, 1);
      {$IF DEFINED(LINUX)}
      sleep(1);
      {$ELSEIF DEFINED(WIN64)}
      sleep(1);
      {$ELSEIF DEFINED(WIN32)}
      sleep(0);
      {$ENDIF}
    until (nc > 0) or (time > timeout);
    if (buf = NAK) or (buf = '?') then
      Message(Format(RECEIVED,[buf, cmd, stmp]))
    else
      stmp := buf+stmp;
  until (buf = ETB) or (buf = NAK) or (time > timeout);

  if time > timeout then begin
    Message(Format(SENDCOMMANDCTIMEOUT,[stmp]));
    exit(ERROR);
  end;

  if not (buf = NAK) then response := trim(stmp);

  if MSmeter.Checked and Reenable_Status then Enable_Status := ENABLED;
  SendCommandC := NOERROR;
end;

// Send a command to EKD500. Assumes mode D (no reply from receiver)
function TEKD.SendCommandD(cmd, par: string): boolean;

var tmp: string;
    i,nc,len: integer;
    ok: boolean;

begin
  Enable_Status := DISABLED;

  tmp := cmd+par;
  len := Length(tmp);

  if MSHOWC.Checked then Message('CmdD: '+tmp);
  DoMessages;

  {$IFDEF TEST_USER_INTERFACE}
  exit(NOERROR);
  {$ENDIF}

  ok := NOERROR;
  for i := 1 to len do begin
    sleep(10+TimeOneCharMs);
    nc := SerWrite(SerPort, tmp[i], 1);
    if nc <> 1 then begin
      Ok := ERROR;
      {$IFDEF TEST}
      Message('SerWrite returned ' + IntToStr(nc));
      {$ENDIF}
    end;
  end;

  if MSmeter.Checked then Enable_status := ENABLED;
  SendCommandD := ok;
end;

// Parse status
procedure TEKD.ParseStatus(Status: string; StatusType: boolean);

var st, sf, sm, sb, sd, se, gc, df: string;
    subtype: integer;
    f: real;
    i: integer;

begin
  st := trim(Status);
  if st = '' then exit;

  if StatusType then begin
    // Mode C status: Mode, bandwidth & frequency or d,gc,se,a,d responses
    if pos('e', st) > 0 then subtype := TYPE_E;
    if pos('se',st) > 0 then subtype := TYPE_SE
    else if pos('e.', st) > 0 then subtype := TYPE_SCAN
    else if pos('d', st) > 0 then subtype := TYPE_D
    else if pos('gc',st) > 0 then subtype := TYPE_GC
    else if pos('c',st) > 0 then subtype := TYPE_C
    else if pos('s',st) > 0 then subtype := TYPE_S
    else if pos('a',st) > 0 then subtype := TYPE_A
    else subtype := TYPE_E;

    case SubType of
      TYPE_SE: begin
        se := copy(st, 4,1);
        RXState.Preselector := StrToInt(se);
        CBSEL.ItemIndex := RXState.Preselector;
      end;
      TYPE_E: begin
        sm := copy(st,1,1);
        sb := copy(st,2,1);
        sf := copy(st,4,8);

        f := 1000*StrToFloat(AdjustDecimalSeparator(sf));
        RXState.Freq_rx := trunc(f);
        SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);

        i := StrToInt(sm);
        RXState.Mode_RX := i;
        CBMode.ItemIndex := i-1;

        i := StrToInt(sb);
        RXState.Filter := i;
        CBBW.ItemIndex := i - 1;
      end;
      TYPE_SCAN: begin
        sm := copy(st,1,1);
        sb := copy(st,2,1);
        sf := copy(st,5,Length(st)-6);

        f := 1000*StrToFloat(AdjustDecimalSeparator(sf));
        RXState.Freq_rx := trunc(f);
        SpinEditUpdateOnly(SPFREQ, RXState.Freq_rx);

        i := StrToInt(sm);
        RXState.Mode_RX := i;
        CBMode.ItemIndex := i-1;

        i := StrToInt(sb);
        RXState.Filter := i;
        CBBW.ItemIndex := i - 1;
      end;
      TYPE_D: begin
        df := copy(st, 3, Length(st));
        RXstate.Freq_step := trunc(1000*StrToFloat(AdjustDecimalSeparator(df)));
        SpinEditUpdateOnly(SPDELTAF, RXstate.Freq_step);
      end;
      TYPE_GC: begin
        gc := copy(st, 7, 1);
        i := StrToInt(gc);
        RXState.AGC := i;
        CBAGC.ItemIndex := i - 1;
        if not MSmeter.Checked then begin
          dBm := 2*StrToInt(copy(st,1,2));
          SetSMeter((dBm div 10) + 1);
          SMLabel.Caption := IntTostr(dbm)+' of 120 ('+CalcSValue(dbm)+')'
        end;
      end;
      TYPE_S, TYPE_C, TYPE_A: ;
      end;
    end else begin
      // Mode B status: Signal strenght & frequency
      // (Currently not used, dbm values are read with the G command)
      sd := copy(Status, 1, 2);
      dBm := 2 * StrToInt(sd);
    end;
    In_Command := FALSE;
end;

// Get status. Assumes Mode C in effect.
procedure TEKD.GetStatus;

var stmp: string = '';

begin
  {$IFDEF TEST_USER_INTERFACE}
  statusC := '12e14250.00';
  exit;
  {$ENDIF}
  stmp:='';
  SendCommandC(Key_SEL, stmp, DONT_REENABLE);
  ParseStatus(stmp,STATUSTYPE_C);
  SendCommandC(Key_E, stmp, DONT_REENABLE);

  SendCommandC(Key_DF, stmp, DONT_REENABLE);
  ParseStatus(stmp,STATUSTYPE_C);
  SendCommandC(Key_E,stmp, DONT_REENABLE);

  SendCommandC(Key_GC, stmp, DONT_REENABLE);
  ParseStatus(stmp,STATUSTYPE_C);
  SendCommandC(Key_E, statusC, REENABLE);

  ParseStatus(statusC,STATUSTYPE_C);
end;

// Update display
procedure TEKD.UpdateDisplay;

var s: string;

begin
  s := trim(statusC);

  if pos('e', s) > 0 then begin
    if pos('se',s) > 0 then begin
      s := '  ' + s;
      while Length(s) < 13 do Insert(' ', s, 5);
    end else begin
      while Length(s) < 12 do Insert(' ', s, 4);
      Insert(' ', s, 3);
    end;
  end else if pos('gc', s) > 0 then begin
    while Length(s) < 13 do Insert(' ', s, 6);
  end else if (pos('d', s) > 0) or (pos('s', s) > 0) or (pos('c',s) > 0) then begin
    s := '  ' + s;
    while Length(s) < 13 do Insert(' ', s, 4);
  end else if pos('a', s) > 0 then begin
    s := '  ' + s;
    while Length(s) < 13 do Insert(' ', s, 5);
  end else begin
    while Length(s) < 13 do Insert(' ', s, 4);
  end;

  PutRXString(s);
  DoMessages;
end;

// Set RX frequency
function TEKD.SetRXFreq: boolean;

var  sf: string;
     rf: real;

begin
  rf := RXState.Freq_rx / 1000;
  str(rf:0:2,sf);
  SetRemoteMode(MODE_D, NOFLUSH);
  SendCommandD(Key_F, sf+Key_E);
  SetRemoteMode(MODE_C, FLUSH_INPUT);
  SetRXFreq := NOERROR;
end;

// Display a message in MSG memo
procedure TEKD.Message(ms: string);

var LineEndingLen, LineLen: integer;
    stmp: string;

begin
  stmp := ms;
  LineEndingLen := Length(LineEnding);
  LineLen := Pos(LineEnding,stmp);
  if LineLen <> 0 then begin
    // Multiline string, divide it into component lines
    // and add them to the MSG memo
    repeat
      LineLen := Pos(LineEnding,stmp);
      if LineLen=0 then
        LineLen := Length(stmp)
      else
        LineLen := LineLen-1;
        MSG.Lines.Add(copy(stmp,1,LineLen));
        Delete(stmp,1,LineLen+LineEndingLen);
    until Length(stmp) = 0;
  end else begin
    // Single line string
    MSG.Lines.Add(stmp);
  end;
  // kludge to make TMemo scroll - not always works but better than nothing
  MSG.VertScrollBar.Position := 10000;
end;

// Clear messages in MSG memo
procedure TEKD.MSGDblClick(Sender: TObject);
begin
  MSG.Lines.Clear;
end;

// Display only RX frequency
procedure TEKD.DispRXf;

var tmp: real;
    s: string = '';

begin
  if RXState.Scan then begin
  // Not needed?
  end else begin
    if RXState.Freq_RX >= 0 then begin
      tmp := RXState.Freq_rx / 1000;
      str(tmp:8:2,s);
    end else begin
      // Ditto
      s := 'CHAN.'+IntToStr(-RXState.Freq_RX);
    end;
  end;
  PUTRXstring(Copy(GetRXString, 1, 5) + s);
  SpinEditUpdateOnly(SPFREQ, RXState.Freq_RX);
end;

// Display only RX mode
procedure TEKD.DispRXm;

var s: string;

begin
  s := GetRXString;
  {$IFDEF TEST}
  Message('statusC='+statusC);
  {$ENDIF}
  PutRXString(copy(statusC, 2, 2)+' E ' + copy(s, 6, 8));
end;

// Update non-button controls
procedure TEKD.UpdateControls;

begin
  with RXState do begin
    SpinEditUpdateOnly(SPFREQ, Freq_rx);
    SpinEditUpdateOnly(SPDELTAF, Freq_step);
    CBMODE.ItemIndex := Mode_RX-1;
    CBBW.ItemIndex := Filter - 1;
    CBAGC.ItemIndex := AGC - 1;
  end;
end;

// Update SpinEdit display only
procedure TEKD.SpinEditUpdateOnly(Sender: TObject; value: integer);

var ASpin: TSpinEdit;

begin
  ASpin := Sender as TSpinEdit;
  Update_Only := TRUE;
  ASpin.Value := Value;
  Update_Only := FALSE;
end;

// Keyboard handler
procedure TEKD.KeyboardHandler(keyp: string);

begin
  SendCommandC(keyp,statusC, REENABLE);
  if ENotRequired then begin
    ParseStatus(statusC, STATUSTYPE_C);
    EnableAllControls;
    LBdBuV.Visible := FALSE;;
    ENotRequired := FALSE;
    In_Command := FALSE;
  end;
  UpdateDisplay;
  SetActiveButtons(B1);
end;


// Create a default configuration if no config file found
procedure TEKD.SetDefaultConfig;

begin
  // default values
  {$IFDEF LINUX}
  SerPortName := '/dev/ttyS0';
  {$ENDIF}

  {$IFDEF WINDOWS}
  SerPortName := 'COM1';
  {$ENDIF}
  SerPortSpeed := 2400;
  MSmeter.Checked := FALSE;
  LEDColorOFF := ClLime;
  LEDColorON := clGreen;
  sLEDColor := 'green';
  MYellow.Checked := FALSE;
  MRed.Checked := FALSE;
  MGreen.Checked := TRUE;
  M2400B.Checked := TRUE;

  // Default starting position and size
  StartX := (Screen.Width - EKD.Width) div 2;
  StartY := (Screen.Height - EKD.Height) div 2;
  StartWidth := 600;
  StartHeight := 358;

  {$IFDEF LINUX}
  if SerPortName = '/dev/ttyS0' then begin
  {$ENDIF}

  {$IFDEF WINDOWS}
  if SerPortName = 'COM1' then begin
  {$ENDIF}
    Mttys0.Checked := TRUE;
    Mttys1.Checked := FALSE;
    MCustom.Checked := FALSE;
  end else begin
  {$IFDEF LINUX}
    if SerPortName = '/dev/ttyS1' then begin
  {$ENDIF}
  {$IFDEF WINDOWS}
    if SerPortName = 'COM2' then begin
  {$ENDIF}
      Mttys0.Checked := FALSE;
      Mttys1.Checked := TRUE;
      MCustom.Checked := FALSE;
    end else begin
      Mttys0.Checked := FALSE;
      Mttys1.Checked := FALSE;
      MCustom.Checked := TRUE;
    end;
  end;

  if SerPortSpeed = 200 then m200b.Checked := TRUE;
  if SerPortSpeed = 300 then m300b.Checked := TRUE;
  if SerPortSpeed = 600 then m600b.Checked := TRUE;
  if SerPortSpeed = 1200 then m1200b.Checked := TRUE;
  if SerPortSpeed = 2400 then m2400b.Checked := TRUE;

  // Default font magnification
  FontMagn := 100;

  HaveConfig := TRUE;
end;

// Save config to disk
procedure TEKD.SaveConfig;

var
  Config: TXMLConfig;                       // variable to document
  w,h,l,t,i: integer;
  s: UnicodeString;

begin
  // Create a XMLConfig document
  Config := TXMLConfig.Create(self);
  Config.Filename := HomeDir+XMLConfigFileName;
  Config.RootName := 'EKD500ControlConfig';

  // Start with an empty file
  Config.Clear;

  // Save config version number
  Config.SetValue('ConfigVersion','1');

  // Save serial port parameters
  Config.OpenKey('CommsSetup');
  Config.SetValue('SerialPortName', UnicodeString(SerPortName));
  Config.SetValue('SerialPortSpeed', SerPortSpeed);
  Config.SetValue('CurrentRXNumber', CurRXNumber);
  Config.SetValue('NumberOfReceivers', NumberOfReceivers);

  for i := 1 to NumberOfReceivers do begin
    s := Format('Receiver%dAddress',[i]);
    Config.SetValue(s, ReceiversAddresses[i]);
  end;
  Config.CloseKey;

  Config.OpenKey('ProgramConfiguration');
  Config.SetValue('ShowSMeter', MSMETER.Checked);
  Config.SetValue('ShowCUCmds', MSHOWC.Checked);
  Config.SetValue('ShowHamlibCmds', MSHOWHAMLIB.Checked);
  Config.SetValue('ShowCUCmds', MSHOWC.Checked);
  Config.SetValue('EnableAllCmds', MENALL.Checked);
  Config.SetValue('LEDColor', UnicodeString(sLEDColor));

  w := ClientWidth;
  h := ClientHeight;
  l := EKD.Left;
  t := EKD.Top;

  // Check w, h, l and t values. Under XP and maybe some other system
  // Top and Left may go negative, so check them also
  if w > Screen.Width then w := Screen.Width;
  if h > Screen.Height then h := Screen.Height;
  if t < 0 then t := 0;
  if l < 0 then l := 0;

  {$IFDEF DEBUG_SAVEDSIZEANDPOS}
  Message(Format('Saved size&pos: %dx%d@%d,%d',[w, h, l,t]));
  DoMessages;
  sleep(5000);
  {$ENDIF}
  Config.SetValue('MainWindowWidth', w);
  Config.SetValue('MainWindowHeight', h);
  Config.SetValue('MainWindowLeft', l);
  Config.SetValue('MainWindowTop', t);

  Config.SetValue('FontMagnification', FontMagn);
  Config.SetValue('MagnifyAllFonts', MMAGALLF.Checked);
  Config.SetValue('TCPServerHost', UnicodeString(EKD500Server.Host));
  Config.SetValue('TCPServerPort', EKD500Server.Port);
  Config.CloseKey;

  Config.Flush;
  Config.Free;
end;

// Read config from disk - XML version
procedure TEKD.ReadConfig;
var s,ConfVersion: string;
    Config: TXMLConfig;
    i: integer;

begin
  s := HomeDir+XMLConfigFileName;
  if FileExists(s) then begin
    // We have an XML config file, read it
    try
      Config := TXMLConfig.Create(self);
      Config.Filename := s;

      // Read config version number
      ConfVersion := Config.GetValue('ConfigVersion','');
      if ConfVersion <> '1' then begin
        ShowMessage(Format(UNSUPPORTEDCONFIG,[ConfVersion]));
      end;
      // Read serial port parameters
      Config.OpenKey('CommsSetup');
      SerPortName := Config.GetValue('SerialPortName', '');
      SerPortSpeed := Config.GetValue('SerialPortSpeed', 2400);
      CurRXNumber := Config.GetValue('CurrentRXNumber', 1);
      NumberOfReceivers := Config.GetValue('NumberOfReceivers', 1);

      for i := 1 to NumberOfReceivers do begin
        s := Format('Receiver%dAddress',[i]);
        ReceiversAddresses[i] := Config.GetValue(s, i);
      end;
      Config.CloseKey;

      Config.OpenKey('ProgramConfiguration');
      MSMETER.Checked := Config.GetValue('ShowSMeter', TRUE);
      MSHOWC.Checked := Config.GetValue('ShowCUCmds', FALSE);
      MSHOWHAMLIB.Checked := Config.GetValue('ShowHamlibCmds', FALSE);
      MSHOWC.Checked := Config.GetValue('ShowCUCmds', FALSE);
      MENALL.Checked := Config.GetValue('EnableAllCmds', FALSE);
      sLEDColor :=Config.GetValue('LEDColor', 'green');
      StartWidth := Config.GetValue('MainWindowWidth', 600);
      StartHeight := Config.GetValue('MainWindowHeight', 306);
      StartX := Config.GetValue('MainWindowLeft', 100);
      StartY := Config.GetValue('MainWindowTop', 100);
      FontMagn := Config.GetValue('FontMagnification', 100);
      MMAGALLF.Checked := Config.GetValue('MagnifyAllFonts', FALSE);
      EKD500Server.Host := Config.GetValue('TCPServerHost', '127.0.0.1');
      EKD500Server.Port := Config.GetValue('TCPServerPort', 4538);
      Config.CloseKey;
      Config.Free;

      // Set serial port menu
      {$IFDEF LINUX}
      {$IFDEF CPUARM}
      if (SerPortName = '/dev/ttyAMA0') then begin
      {$ELSE}
      if (SerPortName = '/dev/ttyS0') then begin
      {$ENDIF}
      {$ENDIF}
      {$IFDEF WINDOWS}
      if SerPortName = '\\.\COM1' then begin
      {$ENDIF}
        Mttys0.Checked := TRUE;
        Mttys1.Checked := FALSE;
        MCustom.Checked := FALSE;
      end else begin
        {$IFDEF LINUX}
        {$IFDEF CPUARM}
        if SerPortName = '/dev/ttyUSB0' then begin
        {$ELSE}
        if SerPortName = '/dev/ttyS1' then begin
        {$ENDIF}
        {$ENDIF}
        {$IFDEF WINDOWS}
        if SerPortName = '\\.\COM2' then begin
        {$ENDIF}
          Mttys0.Checked := FALSE;
          Mttys1.Checked := TRUE;
          MCustom.Checked := FALSE;
        end else begin
          Mttys0.Checked := FALSE;
          Mttys1.Checked := FALSE;
          MCustom.Checked := TRUE;
        end;
      end;

      case SerPortSpeed of
        200: M200B.Checked := TRUE;
        300: M300B.Checked := TRUE;
        600: M600B.Checked := TRUE;
        1200: M1200B.Checked := TRUE;
        2400: M2400B.Checked := TRUE;
        otherwise begin
          M2400B.Checked := TRUE;
          SerPortSpeed := 2400;
        end;
      end;

      if sLEDColor = 'red' then begin
        LEDColorON := clRed;
        MRED.Checked := TRUE;
      end else if sLEDColor = 'green' then begin
        LEDColorON := clLime;
        MGREEN.Checked := TRUE;
      end else if sLEDColor = 'yellow' then begin
        LEDColorON := clYellow;
        MYELLOW.Checked := TRUE;
      end else begin
        // color name invalid, use green
        LEDColorON := clLime;
        MGREEN.Checked := TRUE;
      end;

      if MSmeter.Checked and (SerPortSpeed < 1200) then begin
        Message(SPEEDTOOSLOW);
        MSmeter.Checked := FALSE;
      end;

      MMAG100.Checked := FALSE;
      MMAG110.Checked := FALSE;
      MMAG125.Checked := FALSE;
      MMAG150.Checked := FALSE;
      MMAG175.Checked := FALSE;
      MMAG200.Checked := FALSE;
      case FontMagn of
        100: MMAG100.Checked := TRUE;
        110: MMAG110.Checked := TRUE;
        125: MMAG125.Checked := TRUE;
        150: MMAG150.Checked := TRUE;
        175: MMAG175.Checked := TRUE;
        200: MMAG200.Checked := TRUE;
        otherwise begin
          FontMagn := 100;
          MMAG100.Checked := TRUE;
        end;
      end;

      CBREC.Clear;
      for i := 1 to NumberOfReceivers do begin
         s := Format('Receiver%dAddress',[i]);
         ReceiversAddresses[i] := Config.GetValue(s, i);
         CBREC.Add(IntToStr(i));
      end;
      CurRXAddress := ReceiversAddresses[CurRXNumber];
      CBREC.ItemIndex := CurRXNumber-1;

      RXFreqMod.Font.Color := LedColorOn;
      RXFreqB.Font.Color := LedColorOn;
      RXFreqE.Font.Color := LedColorOn;
      RXFreq10M.Font.Color := LedColorOn;
      RXFreq1M.Font.Color := LedColorOn;
      RXFreq100k.Font.Color := LedColorOn;
      RXFreq10k.Font.Color := LedColorOn;
      RXFreq1k.Font.Color := LedColorOn;
      RXFreqDot.Font.Color := LedColorOn;
      RXFreqDot1.Font.Color := LedColorOn;
      RXFreqL2.Font.Color := LedColorOn;
      RXFreq100H.Font.Color := LedColorOn;
      RXFreq10H.Font.Color := LedColorOn;
      SHext.Brush.Color := LedColorOn;

      SetSMeterColor;
      HaveConfig := TRUE;
    except
      ShowMessage(ERRORREADINGCONF);
    end;
    SetSMeterColor;
    SetFreqColor;
  end else begin
    // No XML config file, check for an old config file
    s := HomeDir+ConfigFileName;
    if FileExists(s) then begin
      // found, show a notice and read it
      ShowMessage(OLDUPDATING);
      ReadConfig_old;
      // And save the new XML version
      SaveConfig;
    end else begin
      // No configuration file, set up and save a default one
      ShowMessage(CONFIGNOTFOUND);
      SetDefaultConfig;
      SaveConfig;
    end;
  end;
end;

// Read config from disk - old text version - for compatibility
procedure TEKD.ReadConfig_old;

var ConfigFile: Text;
    s: string;
    i: integer;

begin
  AssignFile(ConfigFile, HomeDir+ConfigFileName);
  try
    Reset(ConfigFile);

    ReadLn(ConfigFile, SerPortName);

    // Set serial port menu
    {$IFDEF LINUX}
    {$IFDEF CPUARM}
    if (SerPortName = '/dev/ttyAMA0') then begin
    {$ELSE}
    if (SerPortName = '/dev/ttyS0') then begin
    {$ENDIF}
    {$ENDIF}
    {$IFDEF WINDOWS}
    if SerPortName = '\\.\COM1' then begin
    {$ENDIF}
      Mttys0.Checked := TRUE;
      Mttys1.Checked := FALSE;
      MCustom.Checked := FALSE;
    end else begin
      {$IFDEF LINUX}
      {$IFDEF CPUARM}
      if SerPortName = '/dev/ttyUSB0' then begin
      {$ELSE}
      if SerPortName = '/dev/ttyS1' then begin
      {$ENDIF}
      {$ENDIF}
      {$IFDEF WINDOWS}
      if SerPortName = '\\.\COM2' then begin
      {$ENDIF}
        Mttys0.Checked := FALSE;
        Mttys1.Checked := TRUE;
        MCustom.Checked := FALSE;
      end else begin
        Mttys0.Checked := FALSE;
        Mttys1.Checked := FALSE;
        MCustom.Checked := TRUE;
      end;
    end;

    ReadLn(ConfigFile, SerPortSpeed);
    ReadLn(ConfigFile, s);
    MSmeter.Checked := (s='TRUE');
    if MSmeter.Checked and (SerPortSpeed < 1200) then begin
      Message(SPEEDTOOSLOW);
      MSmeter.Checked := FALSE;
    end;
    ReadLn(ConfigFile, s);
    MSHOWC.Checked := (s='TRUE');
    ReadLn(ConfigFile, s);
    MENALL.Checked := (s='TRUE');

    ReadLn(ConfigFile, sLEDColor);
    if sLEDColor = 'red' then begin
      LEDColorON := clRed;
      MRED.Checked := TRUE;
    end;
    if sLEDColor = 'green' then begin
      LEDColorON := clLime;
      MGREEN.Checked := TRUE;
    end;
    if sLEDColor = 'yellow' then begin
      LEDColorON := clYellow;
      MYELLOW.Checked := TRUE;
    end;

    ReadLn(ConfigFile, CurRXNumber);
    ReadLn(ConfigFile, NumberOfReceivers);
    CBREC.Clear;
    for i := 1 to NumberOfReceivers do CBREC.Add(IntToStr(i));

    for i := 1 to NumberOfReceivers do begin
      ReadLn(ConfigFile, ReceiversAddresses[i]);
    end;
    CurRXAddress := ReceiversAddresses[CurRXNumber];
    CBREC.ItemIndex := CurRXNumber-1;

    //Read starting dimension and position
    ReadLn(ConfigFile, StartWidth);
    ReadLn(ConfigFile, StartHeight);
    ReadLn(ConfigFile, StartX);
    ReadLn(ConfigFile, StartY);

    // Read font magnification factor
    ReadLn(ConfigFile, s);
    if not TryStrToInt(s, Fontmagn) then begin
      ShowMessage(OLDSTYLECONF);
      FontMagn := 100;
      SaveConfig;
    end;

    // Read magnify all fonts setting
    ReadLn(ConfigFile, s);
    MMAGALLF.Checked := (s='TRUE');

    CloseFile(ConfigFile);

    MMAG100.Checked := FALSE;
    MMAG110.Checked := FALSE;
    MMAG125.Checked := FALSE;
    MMAG150.Checked := FALSE;
    MMAG175.Checked := FALSE;
    MMAG200.Checked := FALSE;
    case FontMagn of
      100: MMAG100.Checked := TRUE;
      110: MMAG110.Checked := TRUE;
      125: MMAG125.Checked := TRUE;
      150: MMAG150.Checked := TRUE;
      175: MMAG175.Checked := TRUE;
      200: MMAG200.Checked := TRUE;
      otherwise
        MMAG100.Checked := TRUE;
    end;

    RXFreqMod.Font.Color := LedColorOn;
    RXFreqB.Font.Color := LedColorOn;
    RXFreqE.Font.Color := LedColorOn;
    RXFreq10M.Font.Color := LedColorOn;
    RXFreq1M.Font.Color := LedColorOn;
    RXFreq100k.Font.Color := LedColorOn;
    RXFreq10k.Font.Color := LedColorOn;
    RXFreq1k.Font.Color := LedColorOn;
    RXFreqDot.Font.Color := LedColorOn;
    RXFreqDot1.Font.Color := LedColorOn;
    RXFreqL2.Font.Color := LedColorOn;
    RXFreq100H.Font.Color := LedColorOn;
    RXFreq10H.Font.Color := LedColorOn;
    SHext.Brush.Color := LedColorOn;

    SetSMeterColor;
  except
    SetDefaultConfig;
  end;
end;

// Restore state in RXState
procedure TEKD.RestoreState;

var par: string;
    r: real;

begin
  DisableAllButtons;
  // Set mode D (no reply from RX)
  if not SetRemoteMode(Mode_D, TRUE) then exit;

  // Set RX frequency
  SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);
  r := RXState.Freq_rx / 1000;
  str(r:0:2,par);
  SendCommandD(Key_F, par+Key_E);

  // Set frequency step
  if RXState.Freq_step >= 10 then begin
    SpinEditUpdateOnly(SPDELTAF, RXstate.Freq_Step);
    r := RXState.Freq_step / 1000.0;
    str(r:4:2, par);
    if par[4] = '0' then delete(par,4,1);
  end else begin
    // Invalid value, default to 100 Hz
    SpinEditUpdateOnly(SPDELTAF, 100);
    RXState.Freq_step := 100;
    par := '0.1';
  end;
  SendCommandD(Key_DF, par+Key_E);

  // Set mode
  if RXState.Mode_RX in [M_A1..M_F1D] then begin
    par := IntToStr(RXState.Mode_RX);
  end else begin
    // Invalid value, default to A1
    RXState.Mode_RX := M_A1;
    par := '1';
  end;
  CBMode.ItemIndex := RXState.Mode_RX-1;
  if not SendCommandD(Key_MOD, par) then
    Message(COMMANDFAILED);

  // Set filter
  if RXState.Filter in [Filter_150..Filter_3000L] then begin
    CBBW.ItemIndex := RXState.Filter-1;
    par := IntToStr(RXState.Filter);
  end else begin
    // Invalid filter, assume 400 Hz
    RXstate.Filter := Filter_400;
    par := '2';
  end;
  if not SendCommandD(Key_B, par) then
    Message(COMMANDFAILED);

  // Set AGC
  if RXState.AGC in[AVC_FAST..AVC_MAN] then begin
    par := IntToStr(RXstate.AGC);
    CBAGC.ItemIndex := RXState.AGC-1;
  end else begin
    // Invalid value, default to FAST
    RXState.AGC := AVC_FAST;
    par := '1';
  end;
  if not SendCommandD(Key_GC, par+Key_E) then
    Message(COMMANDFAILED);

  // Set RF Filter
  if RXState.Preselector in[SEL_WIDE..SEL_NARROW] then begin
    par := IntToStr(RXstate.Preselector);
    CBSEL.ItemIndex := RXState.Preselector;
  end else begin
     // Invalid value, default to NARROW
     RXState.Preselector := SEL_NARROW;
     par := '1';
  end;
  if not SendCommandD(Key_SEL, par+Key_E) then
    Message(COMMANDFAILED);

  // Status polling
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
    TSMDisp.Enabled := TRUE;
    SMGroupBox.Visible := TRUE;
    In_Command := FALSE;
  end else begin
    TSmeter.Enabled := FALSE;
    TSMDisp.Enabled := FALSE;
    SMGroupBox.Visible := FALSE;
  end;

  // Scan mode
  if RXState.Scan then begin
    //TODO?
  end;

  // Revert to mode C
  SetRemoteMode(Mode_C, FLUSH_INPUT);
  SetRemoteMode(Mode_C, FLUSH_INPUT);

  EnableKeypad;
  EnableCmdKeys;
  EnableCombosAndEdits;

  // Get and display new parameters
  GetStatus;
  UpdateDisplay;
end;

//Create ordered Channels submenus
procedure TEKD.LoadChannelsDir;

var m: TMenuItem;
    n: integer;
    Info: TSearchRec;
    FileList: TStringList;

begin
  FileList := TStringList.Create;
  FileList.Sorted := TRUE;
  if FindFirst(ChannelsDir+'*.dat', faAnyFile, info) = 0 then begin
    repeat
      delete(Info.Name, Length(Info.Name)-3, 4);
      FileList.Add(Info.Name);
    until FindNext(Info) <> 0;
    FindClose(Info);
    MChannels.Clear;
    for n := 0 to FileList.Count-1 do begin
      m := TmenuItem.Create(MChannels);
      m.Caption := FileList.Strings[n];
      m.OnClick := @LoadChannel;
      MChannels.Add(m);
    end;
    FileList.Clear;
    FileList.Free;
  end else begin
    MChannels.Clear;
    m := TmenuItem.Create(MChannels);
    m.Caption := 'No channels defined';
    MChannels.Add(m);
  end;
end;

// Load a channel file
procedure TEKD.LoadChannel(Sender: TObject);

var BandFile: FILE of TRXState;

begin
  In_Command := TRUE;
  AssignFile(BandFile,ChannelsDir+(Sender as TMenuItem).Caption+'.dat');
  try
    Reset(BandFile);
    Read(BandFile,RXState);
    CloseFile(BandFile);
  except
    MsgToShow := CANTOPENCHANFILE;
  end;
  MsgToSHow := RESTORINGSTATE;
  RestoreState;
  UpdateDisplay;
  if not MENALL.Checked then SetActiveButtons(BE);
  MsgToSHow := DONE;
  In_Command := FALSE;
end;

// Disable all controls except EXT and some menus
procedure TEKD.DisableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
      TControl(Components[i]).Enabled := FALSE;
    end;
    if Components[i] is TJLabel then begin
      TJLabel(Components[i]).Font.Color := clGray;;
    end;
  end;

  MLState.Enabled := FALSE;
  MSState.Enabled := FALSE;
  MSCHANN.Enabled := FALSE;
  MRSTATE.Enabled := FALSE;
  MSmeter.Enabled := FALSE;
  MChannels.Enabled := FALSE;
  MPORT.Enabled := TRUE;
  MSPEED.Enabled := TRUE;
  MRXADD.Enabled := TRUE;
  MttyS0.Enabled := TRUE;
  MttyS1.Enabled := TRUE;
  MCustom.Enabled := TRUE;
  M200B.Enabled := TRUE;
  M300B.Enabled := TRUE;
  M600B.Enabled := TRUE;
  M1200B.Enabled := TRUE;
  M2400B.Enabled := TRUE;
  BEXT.Enabled := TRUE;
  GRXFreq.Enabled := TRUE;
//  LBMOD.Enabled := TRUE;
  LBDBuV.Enabled :=FALSE;
end;

// Disable all buttons
procedure TEKD.DisableAllButtons;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TBitBtn then
      TBitBtn(Components[i]).Enabled := FALSE;
    if Components[i] is TJButton then
      TJButton(Components[i]).Enabled := FALSE;
    if Components[i] is TSpinEdit then
      TSpinEdit(Components[i]).Enabled := FALSE;
    if Components[i] is TComboBox then
      TComboBox(Components[i]).Enabled := FALSE;
  end;
end;

// Disable numeric keypad buttons
procedure TEKD.DisableKeypad;

begin
  B0.Enabled := FALSE;
  B1.Enabled := FALSE;
  B2.Enabled := FALSE;
  B3.Enabled := FALSE;
  B4.Enabled := FALSE;
  B5.Enabled := FALSE;
  B6.Enabled := FALSE;
  B7.Enabled := FALSE;
  B8.Enabled := FALSE;
  B9.Enabled := FALSE;
  BDOT.Enabled := FALSE;
  BARROW.Enabled := FALSE;
end;

// Enable numeric keypad buttons
procedure TEKD.EnableKeypad;

begin
  B0.Enabled := TRUE;
  B1.Enabled := TRUE;
  B2.Enabled := TRUE;
  B3.Enabled := TRUE;
  B4.Enabled := TRUE;
  B5.Enabled := TRUE;
  B6.Enabled := TRUE;
  B7.Enabled := TRUE;
  B8.Enabled := TRUE;
  B9.Enabled := TRUE;
  BDOT.Enabled := TRUE;
  BARROW.Enabled := TRUE;
end;

// Enable command keys
procedure TEKD.EnableCmdKeys;

begin
  BEXT.Enabled := TRUE;
  BMOD.Enabled := TRUE;
  BB.Enabled := TRUE;
  BGC.Enabled := TRUE;
  BSEL.Enabled := TRUE;
  BEXTFCT.Enabled := TRUE;
  BSCANFCT.Enabled := TRUE;
  BSCAN.Enabled := TRUE;
  BDELTAF.Enabled := TRUE;
  BF.Enabled := TRUE;
  BCALL99.Enabled := TRUE;
  BCALL98.Enabled := TRUE;
  BCALL.Enabled := TRUE;
  BSTO.Enabled := TRUE;
  BE.Enabled := TRUE;
end;

// Enable Combo and Edit boxes
procedure TEKD.EnableCombosAndEdits;
begin
  SPFREQ.Enabled := TRUE;
  CBREC.Enabled := TRUE;
  SPDELTAF.Enabled := TRUE;
  CBMODE.Enabled := TRUE;
  CBBW.Enabled := TRUE;
  CBAGC.Enabled := TRUE;
  CBSEL.Enabled := TRUE;
end;

// Enable all controls
procedure TEKD.EnableAllControls;

var i: integer;

begin
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
      TControl(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TMenuItem then begin
      TMenuItem(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TSpinEdit then begin
      TSpinEdit(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TComboBox then begin
      TComboBox(Components[i]).Enabled := TRUE;
    end;
    if Components[i] is TLabel then begin
      TLabel(Components[i]).Enabled := TRUE;
      TLabel(Components[i]).Visible := TRUE;
    end;
    if Components[i] is TJLabel then begin
      TJLabel(Components[i]).Enabled := TRUE;
      TJLabel(Components[i]).Font.Color := clBlack;
    end;
    if Components[i] is TShape then begin
      TShape(Components[i]).Visible := TRUE;
    end;
  end;
  if MSmeter.Checked then begin
    TSmeter.Enabled := TRUE;
  end;
  DoMessages;
end;

// Set active/not active buttons & edit state
procedure TEKD.SetActiveButtons(Sender: TObject);

var itag,p: integer;
    f: real;

begin
  if MENALL.Checked then
    // Do not try to disable inactive buttons, so do nothing and return
    exit
  else begin
    if Sender is TJButton then
      itag := (Sender as TJButton).tag
    else
      itag := 0;
    DisableAllButtons;

    case itag of
      T_BSEL: begin
        B0.Enabled := TRUE;
        B1.Enabled := TRUE;
        BE.Enabled := TRUE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BF,T_BDELTAF: begin
        EnableKeypad;
        BE.Enabled := TRUE;
        BARROW.Enabled := TRUE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BMOD,T_BB: begin
        EnableKeypad;
        B0.Enabled := FALSE;
        BE.Enabled := FALSE;
        BARROW.Enabled := FALSE;
        BDOT.Enabled := FALSE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BGC: begin
        B1.Enabled := TRUE;
        B2.Enabled := TRUE;
        B3.Enabled := TRUE;
        B4.Enabled := TRUE;
        B5.Enabled := TRUE;
        BE.Enabled := TRUE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BCALL,T_BSTO: begin
        EnableKeypad;
        BARROW.Enabled := FALSE;
        BDOT.Enabled := FALSE;
        BE.Enabled := TRUE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BSCANFCT: begin
        Enablekeypad;
        BARROW.Enabled := TRUE;
        BDOT.Enabled := TRUE;
        CmdTag := iTag;
        PrevTag := T_NONE;
      end;

      T_BE,T_BARROW: begin
        EnableCmdKeys;
        EnableCombosAndEdits;
        DisableKeypad;
        if PrevTag = T_BGC then begin
          LBMOD.Visible := FALSE;
          LBB.Visible := FALSE;
          LBDBuV.Visible := TRUE;
        end;
        CmdTag := T_NONE;
        PrevTag := T_NONE;
        end;

        T_BDOT: begin
          case CmdTag of
            T_BF,T_BDELTAF: begin
              EnableKeypad;
              BARROW.Enabled := TRUE;
              BE.Enabled := TRUE;
              PrevTag := iTag;
            end;
            T_BSCANFCT: begin
              EnableKeypad;
              BARROW.Enabled := TRUE;
              BDOT.Enabled := TRUE;
            end;
          end;
        end;

        T_B0..T_B9: begin
          case CmdTag of
            T_BF,T_BDELTAF: begin
            BE.Enabled := TRUE;
            if PrevTag = T_BDOT then begin
              p := Pos('.', statusC);
              if (p > 0) and (Length(statusC) - p < 2) then begin
                EnableKeypad;
              end else BE.Enabled := TRUE;
            end else begin
              f := StrToFloat(AdjustDecimalSeparator(copy(statusC, 4, Length(statusC)-3)));
              if f > 2999 then begin
                BDOT.Enabled := TRUE;
                BARROW.Enabled := TRUE;
              end else begin
                EnableKeypad;
              end;
            end;
          end;

          T_BB,T_BMOD: begin
            EnableCmdKeys;
            EnableCombosAndEdits;
          end;

          T_BSEL,T_BGC: begin
            BE.Enabled := TRUE;
          end;

          T_BSTO,T_BCALL: begin
            BE.Enabled := TRUE;
            if Length(statusC) < 5 then EnableKeypad;
          end;

          T_BSCANFCT: begin
            EnableKeypad;
            BE.Enabled := TRUE;
          end;
        end;
      end;
    end;
  end;
end;


//
// Object init & functions
//

// Various initializations at startup
procedure TEKD.FormCreate(Sender: TObject);

var i: integer;
    c: TControl;

begin
  // If installation is severely brokenc(ErrMsg not empty),
  // notify user and close program now.
  if ErrMsg <> '' then begin
    MessageDlg(ERRORMSG, ErrMsg, mtError,[mbOK],'');
    halt;
  end;

  {$IFDEF TEST_USER_INTERFACE}
  EKD.Caption := 'EKD500 Control - USER INTERFACE TEST MODE!';
  {$ELSE}
  EKD.Caption := 'EKD500 Control '+ShortVersion;
  {$ENDIF}

  {$IFDEF DEBUG_WIDTH_AND_HEIGHT}
  ShowMessage('DesignTimePPI='+IntToStr(DesignTimePPI)+ LineEnding+
              'Screen.PixelsPerInch='+IntTostr(Screen.PixelsPerInch));
  ShowMessage('Width='+IntToSTr(Width)+' Heigth='+IntTostr(Height)+LineEnding+
              'ClientWidth='+IntToSTr(CLientWidth)+' CLientHeigth='+IntTostr(ClientHeight));
  {$ENDIF}

  {$IFDEF WINDOWS}
  MttyS0.Caption := 'COM1';
  MttyS1.Caption := 'COM2';
  {$ENDIF}

  // Check X and Y PPI
  PPI_X := Graphics.ScreenInfo.PixelsPerInchX;
  PPI_Y := Graphics.ScreenInfo.PixelsPerInchY;
  if PPI_X <> PPI_Y then ShowMessage(Format(NOSQUAREPIXELS,[PPI_X,PPI_Y]));

  PutRXString('');
  DisableAllControls;

  // Read channels directory
  LoadChannelsDir;

  //Initialize Smeter array
  SMShapes[1] := SMShape1;
  SMShapes[2] := SMShape2;
  SMShapes[3] := SMShape3;
  SMShapes[4] := SMShape4;
  SMShapes[5] := SMShape5;
  SMShapes[6] := SMShape6;
  SMShapes[7] := SMShape7;
  SMShapes[8] := SMShape8;
  SMShapes[9] := SMShape9;
  SMShapes[10] := SMShape10;
  SMShapes[11] := SMShape11;
  SMShapes[12] := SMShape12;
  SMShapes[13] := SMShape13;

  {$IFDEF NOTDEF}
  // Under Windows the vertical placement of objects in a GroupBox is offset
  // by some amount (setting Top=0 makes the object not at the top of GroupBox).
  // The vertical size is also offset by maybe the same amount, so correct
  // these. The initial placement of objects has been done under Linux,
  // where all is as should be.
  // That under Lazarus 1.8.4, maybe it will be fixed in the upcoming 2.0.0.
  for i := 1 to 13 do begin
    SMShapes[i].Top := -10;
  end;
  SMLabel.Top := -5;
  SMGroupBox.Height := 33;
  SMGroupBox.Top := 25;
  {$ENDIF}

  // Read config
  ReadConfig;

  CalibrateDelays;
  PutRXString('');

  // Component & window scaling
  {$IFDEF AUTOSCALE}
  for i := 0 to ComponentCount - 1 do
    if Components[i] is TControl then begin
      c := TControl(Components[i]);
      Xs[i] := c.Left;
      Ys[i] := c.Top;
      Ws[i] := c.Width;
      Hs[i] := c.Height;
      // Font sizes
      if (c is TJLabel) or
         (c is TLabel) or
         (c is TMemo) or
         (c is TSpinEdit) or
         (c is TComboBoxEx) then
        Hf[i] := c.Font.Height;
      if (Components[i] is TJButton) then
        Hf[i] := (c as TJButton).Lcaption.Font.Height;
    end;
  {$IFDEF DEBUG}
  Message('total of '+IntToStr(i)+' components');
  {$ENDIF}

  OH := CLientHeight;
  OW := ClientWidth;
  {$IFDEF DEBUG}
  Message('H='+IntTostr(Height)+' W='+IntTostr(Width));
  Message('OH='+IntToStr(OH)+' OW='+IntToSTR(OW));
  {$ENDIF}
  {$ENDIF}
end;

// Close down program
procedure TEKD.CloseProgram;
begin
  if MSmeter.Checked then begin
    TSmeter.Enabled := FALSE;
    Enable_Status := FALSE;
  end;
  if SerPort > 0 then begin
    SerDrain(SerPort);
    SerClose(SerPort);
  end;
  if HaveConfig then SaveConfig;
  Halt;
end;

procedure TEKD.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  CloseAction := caNone;
  MExitClick(Sender);
end;


// Allows to enter commands using the keyboard
procedure TEKD.FormKeyPress(Sender: TObject; var Key: char);
begin
  if In_Status or SPFREQ.Focused or SPDELTAF.Focused then exit;

  {$IFDEF TEST}
  Message('Key pressed: '+''''+Key+'''');
  {$ENDIF}

  case LowerCase(Key) of
    'f','*': if BF.Enabled then BFClick(BF);
    'd','/': if BDELTAF.Enabled then BDELTAFClick(BDELTAF);
    'm'    : if BMOD.Enabled then BMODCLick(BMOD);
    'b'    : if BB.Enabled then BBClick(BB);
    'g'    : if BGC.Enabled then BGCClick(BGC);
    'v'    : if BSEL.Enabled then BSELClick(BSEL);
    'c'    : if BCALL.Enabled then BCALLCLick(BCALL);
    's'    : if BSTO.Enabled then BSTOClick(BSTO);
    '"'    : if BCALL98.Enabled then BCALL98Click(BCALL98);
    '!'    : if BCALL99.Enabled then BCALL99Click(BCALL99);
    'l'    : if BSCAN.Enabled then BSCANClick(BSCAN);
    'a'    : if BSCANFCT.Enabled then BSCANFCTClick(BSCANFCT);
    'e',#13: if BE.Enabled then BEClick(BE);
    '+','-': while not SendCommandC(Key, statusC, REENABLE) do;
    '0'    : if B0.Enabled then BNumCLick(B0);
    '1'    : if B1.Enabled then BNumCLick(B1);
    '2'    : if B2.Enabled then BNumCLick(B2);
    '3'    : if B3.Enabled then BNumCLick(B3);
    '4'    : if B4.Enabled then BNumCLick(B4);
    '5'    : if B5.Enabled then BNumCLick(B5);
    '6'    : if B6.Enabled then BNumCLick(B6);
    '7'    : if B7.Enabled then BNumCLick(B7);
    '8'    : if B8.Enabled then BNumCLick(B8);
    '9'    : if B9.Enabled then BNumCLick(B9);
    '.'    : if BDOT.Enabled then BDOTClick(BDOT);
    ';'    : if BARROW.Enabled then BARROWClick(BARROW);
  end;
  key := #0;
  UpdateDisplay;
end;

procedure TEKD.FormResize(Sender: TObject);

var i: integer;
    c: TControl;
    NewHeight,NewHeight1,NewHeight2: integer;

begin
  {$IFDEF AUTOSCALE}
  { Here's actual component scaling. }
  for i := 0 to ComponentCount - 1 do begin
    if Components[i] is TControl then begin
      c := TControl(Components[i]);
      c.Left   := Xs[i] * ClientWidth div OW;
      c.Top    := Ys[i] * ClientHeight div OH;
      c.Width  := Ws[i] * ClientWidth div OW;
      c.Height := Hs[i] * ClientHeight div OH;
      {$IFDEF DEBUG}
      Message('h='+IntTostr(c.Height)+' was '+IntTostr(Hs[i]));
      {$ENDIF}
      if c is TComboBoxEx then
        (c as TComboBoxEx).ItemHeight := Hs[i] * ClientHeight div OH;
    end;
    if (Components[i] is TJButton) or
       (Components[i] is TLabel) or
       (Components[i] is TMemo) or
       (Components[i] is TSpinEdit) or
       (Components[i] is TJLabel) or
       (Components[i] is TComboBoxEx)
       then begin
      {$IFDEF DEBUG}
      Message('scaling font');
      {$ENDIF}
      c := TControl(Components[i]);

      // Do not magnify font of labels with tag <> 0 if MMAGALLF not checked
      if not MMAGALLF.Checked and (c is TLabel) and (c.tag <> 0) then begin
        NewHeight1 := Hf[i] * ClientWidth div OW;
        NewHeight2 := Hf[i] * ClientHeight div OH;
        NewHeight := min(abs(NewHeight1), abs(NewHeight2));
      end else begin
        // if MAGALLF checked or tag=0, apply font magnification factor
        NewHeight1 := Hf[i] * FontMagn * ClientWidth div (OW*100);
        NewHeight2 := Hf[i] * FontMagn * ClientHeight div (OH*100);
        NewHeight := min(abs(NewHeight1), abs(NewHeight2));
      end;
      {$IFDEF DEBUG_FONTSCALE}
      if c is TLabel then
        Message('h1='+IntToStr(NewHeight1)+
                'h2='+IntToStr(NewHeight2)+
                'h='+IntToStr(NewHeight));
      {$ENDIF}
      if c is TJButton then
        (c as TJButton).LCaption.Font.Height := NewHeight
      else
        c.Font.Height := NewHeight;
    end;
    {$IFDEF DEBUG}
    Message('s1='+inttostr(NewSize1)+' s2='+ inttostr(newsize2));
    Message('Width='+IntToSTr(Width)+', Height='+IntToStr(Height));
    Message('ClientWidth='+IntToSTr(CLientWidth)+', ClientHeight='+IntToStr(ClientHeight));
    {$ENDIF}
  end;
  {$ENDIF}
end;

procedure TEKD.FormShow(Sender: TObject);

begin
  // Set last saved size
  EKD.Width := StartWidth;
  EKD.Height := StartHeight;

  {$IFDEF DEBUG_STARTSIZE&POS}
  Message(Format('At formshow start: %dx%d@%d,%d',[EKD.Width, EKD.Height, EKD.Left, EKD.Top]));
  Message(Format('Required size&pos: %dx%d@%d,%d',[StartWidth, StartHeight, StartX,StartY]));
  {$ENDIF}

   // Resize window if larger than screen size
  if EKD.Width > Screen.Width then EKD.Width := Screen.Width;
  if EKD.Height > Screen.Height then EKD.Height := Screen.Height;
  // Adapt main window to the screen height
  if EKD.Height > Screen.Height then EKD.Height := Screen.Height;
  if EKD.Width > Screen.Width then EKD.Width := Screen.Width;

  //Put window in the saved position

  if StartX <= Screen.Width - EKD.Width then
    EKD.Left := StartX
  else
    EKD.Left := Screen.Width - EKD.Width;

  if StartY <= Screen.Height - EKD.Height then
    EKD.Top := StartY
  else
    EKD.Top := Screen.Height - EKD.Height;


  EKD.Width := StartWidth;
  EKD.Height := StartHeight;

  // Apply font magnification factor
  FontMagnify;

  {$IFDEF TEST_USER_INTERFACE}
  GetStatus;
  UpdateDisplay;
  UpdateControls;
  SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);
  {$ENDIF}

  // Set RX# ComboBox to void
  CBREC.ItemIndex := -1;

  {$IFDEF DEBUG_STARTSIZE&POS}
  Message(Format('At formshow end: %dx%d@%d,%d',[EKD.Width, EKD.Height, EKD.Left, EKD.Top]));
  {$ENDIF}
end;

//
// Menu handling
//

// "File" menu handling

// "Load state"
procedure TEKD.MLSTATEClick(Sender: TObject);

var StateFile: FILE of TRXState;
    tmpRXState: TRXState;

begin
  OpenDialog1.InitialDir := StateDir;
  if OpenDialog1.Execute then begin
    AssignFile(StateFile,OpenDialog1.Filename);
    try
      Reset(StateFile);
      Read(StateFile,tmpRXState);
      CloseFile(StateFile);
    except
      Message(Format(CANTOPENSTATEFILE,[Opendialog1.Filename]));
      exit;
    end;
    RXState := tmpRXState;
    RestoreState;
    UpdateControls;
    if not MENALL.Checked then SetActiveButtons(BE);
  end;
end;

// "Save state" or "Save channel"
procedure TEKD.MSSTATEClick(Sender: TObject);

var StateFile: file of TRXState;
    SModeRX: string = '';
    sMtmp,sFtmp: string;

begin
  case RXState.Mode_RX of
    M_A1:  SModeRX := 'A1';
    M_A3:  SModeRX := 'A3';
    M_R3:  SmodeRX := 'R3';
    M_J3:  SModeRX := 'J3';
    M_Br8: SModeRX := 'Br8';
    M_B8:  SModeRX := 'B8';
    M_F0:  SModeRX := 'F0';
    M_F1U: SModeRX := 'F_-_';
    M_F1D: SModeRX := 'F-_-';
  end;

  sMtmp := sModeRX;

  if RXState.Freq_RX < 0 then begin
    sMtmp := '';
    sFtmp := GetRXString;
  end;

  sFtmp := trim(copy(GetRXString, 5, 8));

  if Sender = MSSTATE then SaveDialog1.InitialDir := StateDir;
  if Sender = MSCHANN then SaveDialog1.InitialDir := ChannelsDir;
  SaveDialog1.FileName := sFtmp + '-' + sMtmp + '.dat';
  if SaveDialog1.Execute then begin
    AssignFile(StateFile,SaveDialog1.Filename);
    try
      Rewrite(StateFile);
      Write(StateFile,RXState);
      CloseFile(StateFile);
    except
      Message(Format(CANTWRITESTATEFILE, [Savedialog1.Filename]));
    end;
  end;
  if Sender = MSCHANN then LoadChannelsDir;
end;


// "Exit"
procedure TEKD.MExitClick(Sender: TObject);
begin
  if MessageDlg(CLPROGRAM,REALLYEXIT,mtConfirmation,
               [mbYES, mbNO],0) = mrYES then CloseProgram;
end;

// "Restore current state"
procedure TEKD.MRSTATEClick(Sender: TObject);

begin
  RestoreState;
  EnableAllControls;
  SetActiveButtons(BE);
  UpdateControls;
end;

procedure TEKD.MRXADDClick(Sender: TObject);

var i: integer;

begin
  urxaddr.FADDR.SGRECADD.RowCount := NumberOfReceivers+1;
  for i := 1 to NumberOfReceivers do begin
    urxaddr.FADDR.SGRECADD.Cells[0,i] := IntToStr(i);
    urxaddr.FADDR.SGRECADD.Cells[1,i] := IntToStr(ReceiversAddresses[i]);
  end;
  FADDR.Show;
end;

//
// "Options" menu handling
//

// "RS232 port"

// "ttyS0"
procedure TEKD.Mttys0Click(Sender: TObject);

begin
  {$IFDEF LINUX}
  SerPortName := '/dev/ttyS0';
  {$ENDIF}

  {$IFDEF WINDOWS}
  SerPortName := '\\.\COM1';
  {$ENDIF}
  SaveConfig;
end;

// "ttyS1"
procedure TEKD.Mttys1Click(Sender: TObject);

begin
  {$IFDEF LINUX}
  SerPortName := '/dev/ttyS1';
  {$ENDIF}

  {$IFDEF WINDOWS}
  SerPortName := '\\.\COM2';
  {$ENDIF}
  SaveConfig;
end;

// "Custom..."
procedure TEKD.MCustomClick(Sender: TObject);
begin
  FSerport.Show;
  FSerport.Eser.Text := SerPortName;
end;

// "RS232 speed"

// 200 baud
procedure TEKD.M200BClick(Sender: TObject);
begin
  SerPortSpeed := 200;
  SaveConfig;
  CalibrateDelays;
end;

// "300 baud"
procedure TEKD.M300BClick(Sender: TObject);
begin
  SerPortSpeed := 300;
  SaveConfig;
  CalibrateDelays;
end;

// "600 baud"
procedure TEKD.M600BClick(Sender: TObject);
begin
  SerPortSpeed := 600;
  SaveConfig;
  CalibrateDelays;
end;

// "1200 baud"
procedure TEKD.M1200BClick(Sender: TObject);
begin
  SerPortSpeed := 1200;
  SaveConfig;
  CalibrateDelays;
end;

// "2400 baud"
procedure TEKD.M2400BClick(Sender: TObject);
begin
  SerPortSpeed := 2400;
  SaveConfig;
  CalibrateDelays;
end;

// "S-meter reading"
procedure TEKD.MSmeterClick(Sender: TObject);
begin
  MSmeter.Checked := not MSmeter.checked;
  SaveConfig;
  UpdateControls;
  if MSmeter.Checked then begin
    if SerPortSpeed < 1200 then begin
      Message(SPEEDTOOSLOW);
      MSmeter.Checked := FALSE;
      exit;
    end;
    TSmeter.Enabled := TRUE;
    TSMDisp.Enabled := TRUE;
    Enable_Status := TRUE;
    SMGroupBox.Visible := TRUE;
  end else begin
    TSmeter.Enabled := FALSE;
    TSMDisp.Enabled := FALSE;
    Enable_Status := FALSE;
    SMGroupBox.Visible := FALSE;
  end;
end;

// "LED Color"

// "Green"
procedure TEKD.MGreenClick(Sender: TObject);

begin
  MRed.Checked := FALSE;
  MGreen.Checked := TRUE;
  MYellow.Checked := FALSE;
  LEDCOlorOFF := clGreen;
  LEDColorON := clLime;
  sLEDColor := 'green';
  RXFreqMod.Font.Color := LedColorOn;
  RXFreqB.Font.Color := LedColorOn;
  RXFreqDOT1.Font.Color := LedCOlorOn;
  RXFreqL2.Font.Color := LedCOlorOn;;
  RXFreqE.Font.Color := LedColorOn;
  RXFreq10M.Font.Color := LedColorOn;
  RXFreq1M.Font.Color := LedColorOn;
  RXFreq100k.Font.Color := LedColorOn;
  RXFreq10k.Font.Color := LedColorOn;
  RXFreq1k.Font.Color := LedColorOn;
  RXFreqDot.Font.Color := LedColorOn;
  RXFreq100H.Font.Color := LedColorOn;
  RXFreq10H.Font.Color := LedColorOn;
  SHext.Brush.Color := LedColorOn;
  SetSMeterColor;
  SaveConfig;
  UpdateControls;
end;

// "Yellow"
procedure TEKD.MYellowClick(Sender: TObject);
begin
  MRed.Checked := FALSE;
  MGreen.Checked := FALSE;
  MYellow.Checked := TRUE;
  LEDCOlorOFF := clOlive;
  LEDColorON := clYellow;
  sLEDColor := 'yellow';
  RXFreqMod.Font.Color := LedColorOn;
  RXFreqB.Font.Color := LedColorOn;
  RXFreqDOT1.Font.Color := LedCOlorOn;
  RXFreqL2.Font.Color := LedCOlorOn;;
  RXFreqE.Font.Color := LedColorOn;
  RXFreq10M.Font.Color := LedColorOn;
  RXFreq1M.Font.Color := LedColorOn;
  RXFreq100k.Font.Color := LedColorOn;
  RXFreq10k.Font.Color := LedColorOn;
  RXFreq1k.Font.Color := LedColorOn;
  RXFreqDot.Font.Color := LedColorOn;
  RXFreq100H.Font.Color := LedColorOn;
  RXFreq10H.Font.Color := LedColorOn;
  SHext.Brush.Color := LedColorOn;
  SetSMeterColor;
  SaveConfig;
  UpdateControls;
end;

// "Red"
procedure TEKD.MRedClick(Sender: TObject);

begin
  MRed.Checked := TRUE;
  MGreen.Checked := FALSE;
  MYellow.Checked := FALSE;
  LEDCOlorOFF := clMaroon;
  LEDColorON := clRed;
  sLEDColor := 'red';
  RXFreqMod.Font.Color := LedColorOn;
  RXFreqB.Font.Color := LedColorOn;
  RXFreqDOT1.Font.Color := LedCOlorOn;
  RXFreqL2.Font.Color := LedCOlorOn;;
  RXFreqE.Font.Color := LedColorOn;
  RXFreq10M.Font.Color := LedColorOn;
  RXFreq1M.Font.Color := LedColorOn;
  RXFreq100k.Font.Color := LedColorOn;
  RXFreq10k.Font.Color := LedColorOn;
  RXFreq1k.Font.Color := LedColorOn;
  RXFreqDot.Font.Color := LedColorOn;
  RXFreq100H.Font.Color := LedColorOn;
  RXFreq10H.Font.Color := LedColorOn;
  SHext.Brush.Color := LedColorOn;
  SetSMeterColor;
  SaveConfig;
  UpdateControls;
end;

// "Enable all controls"
procedure TEKD.MENALLClick(Sender: TObject);

begin
  MENALL.Checked := not MENALL.Checked;
  SaveConfig;
  if SerPort <> 0 then begin
    if MENALL.Checked then EnableAllControls else SetActiveButtons(BE);
    UpdateControls;
  end;
end;

// "Band scan window" menu item
procedure TEKD.MBSCANWClick(Sender: TObject);

begin
  FSCAN.Show;
end;

// "Font magnification" submenus
procedure TEKD.MFontMagnClick(Sender: TObject);

begin
  FontMagn := (Sender as TMenuItem).tag;
  SaveConfig;
  FontMagnify;
end;

// "Magnify all fonts" option
procedure TEKD.MMAGALLFClick(Sender: TObject);
begin
  FontMagnify;
  SaveConfig;
end;

// Channel manager
procedure TEKD.MMANCHClick(Sender: TObject);

begin
  UMGR.Caption := CHANMGR;
  UMGR.TFLB.Directory := ChannelsDir;
  UMGR.TFLB.UpdateFileList;
  UMGR.Tag := 1;
  UMGR.SPMOVE2CH.Visible := FALSE;
  UMGR.SPMOVE2ST.Visible := TRUE;
  UMGR.Show;
end;

// State manager
procedure TEKD.MMANSTClick(Sender: TObject);
begin
  UMGR.Caption := STATEMGR;
  UMGR.TFLB.Directory := StateDir;
  UMGR.TFLB.UpdateFileList;
  UMGR.Tag := 0;
  UMGR.SPMOVE2CH.Visible := TRUE;
  UMGR.SPMOVE2ST.Visible := FALSE;
  UMGR.Show;
end;

// Number of receivers
procedure TEKD.MNUMRXClick(Sender: TObject);
begin
  FNRX.SPNRX.Value := NumberOfReceivers;
  FNRX.Show;
end;

// Show commands
procedure TEKD.MSHOWCClick(Sender: TObject);
begin
  SaveConfig;
end;

procedure TEKD.MSHOWHAMLIBClick(Sender: TObject);
begin
  SaveConfig;
end;


//
// "Help" menu handling
//

// "About"
procedure TEKD.MABOUTClick(Sender: TObject);

begin
  FAbout.Show;
end;

procedure TEKD.MADDRPClick(Sender: TObject);
begin
  FTCP.EDADDPORT.Text := EKD500Server.Host+':'+IntToStr(EKD500Server.Port);
  FTCP.Show;
end;

// "Manual"
procedure TEKD.MMANClick(Sender: TObject);

begin
  FMAN.Show;
end;

// "GNU License"
procedure TEKD.MGNULICClick(Sender: TObject);
begin
  FGNU.Show;
end;

//
// Buttons handling (block 1 row 1)
//

// EXT
procedure TEKD.BEXTClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  if SerPort > 0 then begin
    if MSmeter.Checked then begin
      TSmeter.Enabled := FALSE;
      Enable_Status := FALSE;
    end;
    CloseRemote;
    SerFlush(SerPort);
    SerClose(SerPort);

    SerPort := 0;
    SaveConfig;
    sMode_RX := '';
    PutRXString('');
    sMode_TX := '';
    TSmeter.Enabled := FALSE;
    TSMDisp.Enabled := FALSE;
    RXFreq10H.Visible := FALSE;
    RXFreq100H.Visible := FALSE;
    RXFreq1k.Visible := FALSE;
    RXFreq10k.Visible := FALSE;
    RXFreq10H.Visible := FALSE;
    RXFreq100k.Visible := FALSE;
    RXFreq1M.Visible := FALSE;
    RXFreq10M.Visible := FALSE;
    RXFreqE.Visible := FALSE;
    RXFreqDot.Visible := FALSE;
    RXFreqMod.Visible := FALSE;
    RXFreqB.Visible := FALSE;
    MBSCANW. Enabled := FALSE;
    SetSMeter(0);
    DisableAllControls;
    SHEXT.Brush.Color := clSilver;
    CBRec.ItemIndex := -1;
    CBMode.ItemIndex := -1;
    CBBW.ItemIndex := -1;
    CBAGC.ItemIndex := -1;
    CBSel.ItemIndex := -1;
    EKD500Server.Disconnect(TRUE);
    NumTCPConnections := 0;
    DoMessages;
  end else begin
    MsgToShow := STARTINGPROGRAM;
    DoMessages;
    {$IFNDEF TEST_USER_INTERFACE}
    SerPort := SerOpen(SerPortName);
    {$ELSE}
    SerPort:= 1;
    {$ENDIF}
    if SerPort > 0 then begin
      SerSetParams(SerPort,SerPortSpeed,7,EvenParity,2,[]);
      SerSetDTR(SerPort, TRUE);
      SerSetRTS(SerPort, TRUE);
      if OpenRemote then begin
        SHEXT.Brush.Color := LedColorOn;
        EnableAllControls;
        LBDBuV.Visible := FALSE;
        if not MENALL.Checked then DisableKeypad;
        DoMessages;
        SetRemoteMode(Mode_C, FLUSH_INPUT);
        FontMagnify;
        UpdateDisplay;
        MBSCANW. Enabled := TRUE;
        if MSmeter.Checked then begin
          TSMDisp.Enabled := TRUE;
          TSmeter.Enabled := TRUE;
          Enable_Status := TRUE;
          SMGroupBox.Visible := TRUE;
        end;
        // Set RX# ComboBox to active receiver
        CBREC.ItemIndex := CurRXnumber-1;

        // Start rigctl server
        if not EKD500Server.Listen(EKD500Server.Port,EKD500Server.Host) then
          Message(LISTENFAILED);
        end else begin
          SerDrain(SerPort);
          SerClose(SerPort);
          SerPort := 0;
          if MessageDlg(ERRORMSG, ERRORINIT,
                        mtConfirmation, [mbYES, mbNO],0) = mrNO then begin
            CloseProgram;
          end else begin
            In_Command := FALSE;
            BEXTClick(self);
          end;
        end;
      end else begin
        if MessageDlg(ERRORMSG,Format(ERROROPENCOM,[SerPortName]),
                      mtConfirmation, [mbYES, mbNO],0) = mrNO then begin
          CloseProgram;
        end else begin
          In_Command := FALSE;
          BEXTClick(self);
        end;
      end
  end;
  MsgToShow := READY;
  In_Command := FALSE;
end;

// MOD
procedure TEKD.BMODClick(Sender: TObject);
begin
    if In_Command then exit;
    In_Command := TRUE;
    ENotRequired := TRUE;
    while not SendCommandC(Key_MOD, statusC, DONT_REENABLE) do;
    Message(ENTERMODE);
    SetActiveButtons(Sender);
end;

// B
procedure TEKD.BBClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  ENotRequired := TRUE;
  while not SendCommandC(Key_B, statusC, DONT_REENABLE) do;
  Message(ENTERBANDWIDTH);
  SetActiveButtons(Sender);
end;

// GC
procedure TEKD.BGCClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_GC, statusC, DONT_REENABLE) do;
  Message(ENTERAGC);
  UpdateDisplay;
  SetActiveButtons(Sender);
  LBMOD.Visible := FALSE;
  LBB.Visible := FALSE;
  LBDBuV.Visible := TRUE;
end;

//
// Buttons handling (block 1 row 2)
//

// EXT FCT
procedure TEKD.BEXTFCTClick(Sender: TObject);

var s: string;

begin
  Message(NOREMOTECMD);
  Str(CurRXAddress:2, s);
  s := '    b'+IntToStr(SerPortSpeed)+'E.'+s;
  PutRXString(s);
  DoMessages;
  sleep(1000);
  UpdateDisplay;
end;

// SCAN FCT
procedure TEKD.BSCANFCTClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_SCANFCT, statusC, DONT_REENABLE) do;
  Message(ENTERSCANPARAMS);
  UpdateDisplay;
  SetActiveButtons(Sender);
end;

// DELTAF
procedure TEKD.BDELTAFClick(Sender: TObject);

begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_DF, statusC, DONT_REENABLE) do;
  Message(ENTERFSTEP);
  UpdateDisplay;
  SetActiveButtons(Sender);
end;

// F
procedure TEKD.BFClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_F, statusC, DONT_REENABLE) do;
  Message(ENTERF);
  SetActiveButtons(Sender);
end;

// SCAN
procedure TEKD.BSCANClick(Sender: TObject);

var buf: char = chr(0);
    nc,l: integer;
    stmp: string;

begin
  if In_Command and not RXState.Scan then exit;
  In_Command := TRUE;
  DisableAllButtons;
  BSCAN.Enabled := TRUE;
  TSmeter.Enabled := FALSE;
  RXState.Scan := not RXState.Scan;
  {$IFDEF LINUX}
  if RXState.Scan then
    BSCAN.LCaption.Caption := 'SCAN'+LineEnding+'STOP'
  else
    BSCAN.LCaption.Caption := 'SCAN';
  {$ENDIF}
  while not SendCommandC(Key_SCAN, statusC, DONT_REENABLE) do;
  while RXState.Scan do begin
    stmp := '';
    repeat
      timeout := time + READTIMEOUT;
      repeat
        nc := SerRead(SerPort, buf, 1);
        {$IFDEF LINUX}
        sleep(1);
        {$ENDIF}
        {$IFDEF WINDOWS}
        // under windows minimum sleep is 15ms
        sleep(0);
        {$ENDIF}
      until (nc > 0) or (time > timeout);
      if (buf = NAK) or (buf = '?') then
        Message(Format(RECVD,[buf]))
      else
        stmp := buf+stmp;
    until (buf = ETB) or (buf = NAK) or (time > timeout);
    stmp := trim(stmp);
    l := Length(stmp);
    {$IFDEF TEST}
    Message(stmp);
    {$ENDIF}
    if l > 1 then begin
      ParseStatus(stmp, STATUSTYPE_C);
      stmp := copy(stmp,1, 2)+' e.'+copy(stmp,5,l-5);
      while Length(stmp) < 13 do Insert(' ', stmp, 6);
      PutRXString(stmp);
    end;
    DoMessages;
  end;
  while SerRead(SerPort, buf, 1) > 0 do DoMessages;
  EnableCmdKeys;
  EnableCombosAndEdits;
  GetStatus;
  UpdateDisplay;
  TSmeter.Enabled := TRUE;
  In_Command := FALSE;
end;

// SEL
procedure TEKD.BSELClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_SEL, statusC, DONT_REENABLE) do;
  Message(ENTERPRESTYPE);
  UpdateDisplay;
  SetActiveButtons(Sender);
end;


//
// Buttons handling (block 2 row 1)
//

// CALL 99
procedure TEKD.BCALL99Click(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_CALL99, statusC, REENABLE) do;
  GetStatus;
  UpdateDisplay;
  In_Command := FALSE;
end;

// CALL 98
procedure TEKD.BCALL98Click(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_CALL98, statusC, REENABLE) do;
  GetStatus;
  UpdateDisplay;
  In_Command := FALSE;
end;

// CALL
procedure TEKD.BCALLClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_CALL, statusC, DONT_REENABLE) do;
  Message(ENTERCHANNEL);
  UpdateDisplay;
  SetActiveButtons(Sender);
end;

// STO
procedure TEKD.BSTOClick(Sender: TObject);
begin
  if In_Command then exit;
  In_Command := TRUE;
  while not SendCommandC(Key_STO, statusC, DONT_REENABLE) do;
  Message(ENTERCHANNEL);
  SetActiveButtons(Sender);
  UpdateDisplay;
end;

// E
procedure TEKD.BEClick(Sender: TObject);
begin
  In_Command := TRUE;
  ParseStatus(statusC, STATUSTYPE_C);
  while not SendCommandC(Key_E, statusC, REENABLE) do;
  ParseStatus(statusC, STATUSTYPE_C);
  SetActiveButtons(Sender);
  LBDBuV.Visible := FALSE;
  LBMOD.Visible := TRUE;
  LBB.Visible := TRUE;
  UpdateDisplay;
  In_Command := FALSE;
end;


//
// Buttons handling (block 3: numeric keypad)
//

// Numeric key pressed
procedure TEKD.BNumClick(Sender: TObject);
begin
  KeyboardHandler(chr((Sender as TJButton).tag+$30));
end;

// .
procedure TEKD.BDOTClick(Sender: TObject);
begin
  In_Command := TRUE;
  while not SendCommandC(Key_DOT, statusC, DONT_REENABLE) do;
  UpdateDisplay;
  SetActiveButtons(Sender);
end;

// ->
procedure TEKD.BARROWClick(Sender: TObject);
begin
  In_Command := TRUE;
  ParseStatus(statusC, STATUSTYPE_C);
  while not SendCommandC(Key_ARROW, statusC, DONT_REENABLE) do;
  ParseStatus(statusC, STATUSTYPE_C);
  UpdateDisplay;
  SetActiveButtons(Sender);
  In_Command := FALSE;
end;


//
// ComboBox & SpinEdit
//

// AGC ComboBox
procedure TEKD.CBAGCChange(Sender: TObject);
begin
  In_Command := TRUE;
  DisableAllButtons;
  while not SendCommandC(Key_GC, statusC, DONT_REENABLE) do;
  while not SendCommandC(IntToStr(CBAGC.ItemIndex+1), statusC, DONT_REENABLE) do;
  while not SendCommandC(Key_E, statusC, REENABLE) do;
  RXstate.AGC := CBAGC.ItemIndex+1;
  EnableCmdKeys;
  EnableCombosAndEdits;
  In_Command := FALSE;
end;

// BANDWIDTH ComboBox
procedure TEKD.CBBWChange(Sender: TObject);
begin
  In_Command := TRUE;
  DisableAllButtons;
  while not SendCommandC(Key_B, statusC, DONT_REENABLE) do;
  while not SendCommandC(IntToStr(CBBW.ItemIndex+1), statusC, REENABLE) do;
  RXState.Filter := CBBW.ItemIndex+1;
  // No E required after B command
  EnableCmdKeys;
  EnableCombosAndEdits;
  UpdateDisplay;
  In_Command := FALSE;
end;

// MODE ComboBox
procedure TEKD.CBMODEChange(Sender: TObject);

begin
  In_Command := TRUE;
  DisableAllButtons;
  while not SendCommandC(Key_MOD, statusC, DONT_REENABLE) do;
  while not SendCommandC(IntToStr(CBMODE.ItemIndex+1), statusC, REENABLE) do;
  RXstate.Mode_RX := CBMODE.ItemIndex+1;
  // No E required after M command
  GetStatus;
  UpdateDisplay;
  EnableCmdKeys;
  EnableCombosAndEdits;
  In_Command := FALSE;
end;

// Receiver selection ComboBox
procedure TEKD.CBRECChange(Sender: TObject);
begin
  If In_Command then exit;
  CurRXAddress := ReceiversAddresses[CBREC.ItemIndex+1];
  CurRXNumber := CBREC.ItemIndex+1;
  MsgToShow := Format(SWITCHINGTORX,[CurRXNumber]);
  DoMessages;
  In_Command := TRUE;
  if not OpenRemote then begin
    Message(Format(NORXAT,[CBREC.ItemIndex+1,CurRXAddress]));
    if Enable_Status then begin
      Message(DISABLINGSTATUS);
      Enable_Status := FALSE;
    end;
  end;
  MsgToShow := DONE;
  DoMessages;
  In_Command := FALSE;
end;

// Preselector ComboBox
procedure TEKD.CBSELChange(Sender: TObject);
begin
  In_Command := TRUE;
  DisableAllButtons;
  while not SendCommandC(Key_SEL, statusC, DONT_REENABLE) do;
  while not SendCommandC(IntToStr(CBSEL.ItemIndex), statusC, DONT_REENABLE) do;
  while not SendCommandC(Key_E, statusC, REENABLE) do;
  RXState.Preselector := CBSEL.ItemIndex;
  EnableCmdKeys;
  EnableCombosAndEdits;
  In_Command := FALSE;
end;

// DELTAF SpinEdit
procedure TEKD.SPDELTAFChange(Sender: TObject);

var rTmp: real;
    s: string;

begin
  {$IFDEF LINUX}
  if Update_Only then exit;
  {$ENDIF}
  {$IFDEF WINDOWS}
  // Under Windows the OnChange event fires also when it shouldnt
  if Update_Only or (RXState.Freq_Step = SPDELTAF.Value) then exit;
  {$ENDIF}
  RXState.Freq_step := SPDELTAF.Value;
  rTmp := RXState.Freq_step / 1000.0;
  str(rTmp:4:2, s);
  if s[4] = '0' then delete(s,4,1);
  SetRemoteMode(MODE_D, NOFLUSH);
  SendCommandD(Key_DF, s+Key_E);
  SetRemoteMode(MODE_C, FLUSH_INPUT);
end;

// Frequency SpinEdit
procedure TEKD.SPFREQChange(Sender: TObject);
begin
  if SerPort=0 then exit;
  {$IFDEF LINUX}
  if Update_Only then exit;
  {$ENDIF}
  {$IFDEF WINDOWS}
  // Under Windows the OnChange event fires also when it shouldnt
  if Update_Only or (RXState.Freq_RX = SPFREQ.Value) then exit;
  {$ENDIF}
  RXState.Freq_RX := SPFREQ.Value;
  SetRXFreq;
  DispRXF;
end;

// Alternative frequency setting way: left/right click on the frequency figures
// Clicking on the decimal point rewrites displayed frequency
procedure TEKD.RXfreqMouseUp(Sender: TObject; Button: TMouseButton;
        Shift: TShiftState; X, Y: Integer);

var step: integer;

begin
    if In_Command or In_RXFreqMouseUp then exit;

    {$IFDEF TEST}
    Message('X='+IntToStr(X)+' Y='+IntTostr(Y));
    {$ENDIF}

    In_RXFreqMouseUp := TRUE;
    step := (Sender as TLabel).Tag;
    // The . label has tag = -1 to recognise it
    // (the tag 0 is reserved for labels to magnify)
    if step < 0 then step := 0;
    if Button = mbLeft then begin
        if RXState.Freq_rx - step >= MINRXF then
	    RXState.Freq_rx := RXState.Freq_rx-step;
	end;
    if Button = mbRight then begin
        if RXState.Freq_rx + step <= MAXRXF then
		RXState.Freq_rx := RXState.Freq_rx+step;
	end;
    if not SetRXFreq then
        Message(COMMANDFAILED);
    DispRXf;

    In_RXFreqMouseUp := FALSE;
end;

// Allows to use mouse wheel for tuning

// Mouse wheel over frequency display
procedure TEKD.RXFreqMouseWheel(Sender: TObject; Shift: TShiftState;
    WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

var step: integer = -1;

begin
  if In_Wheel or In_Status then exit;
  In_Wheel := TRUE;

  // Set step size
  Step := (Sender as TLabel).Tag;

  if WheelDelta > 0 then
    if RXState.Freq_rx + step <= MAXRXF then
      RXState.Freq_rx := RXState.Freq_rx+step;
  if WheelDelta < 0 then
    if RXState.Freq_rx - step >= MINRXF then
      RXState.Freq_rx := RXState.Freq_rx-step;
  if not SetRXFreq then
    Message(COMMANDFAILED);
  DispRXf;
  In_Wheel := FALSE;
  Handled := TRUE;
end;

// Mouse wheel anywhere else
procedure TEKD.FormMouseWheel(Sender: TObject; Shift: TShiftState;
        WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);

begin
  if In_Wheel or In_Status or (SerPort = 0) then exit;
  In_Wheel := TRUE;
  {$IFDEF TEST}
  Message('X='+IntToStr(MousePos.X)+' Y='+IntTostr(MousePos.Y));
  {$ENDIF}
  if WheelDelta > 0 then begin
    if RXState.Freq_rx+RXState.Freq_step <= MAXRXF then begin
      if SendCommandC(Tune_Up, statusC, REENABLE) then begin
        RXState.Freq_rx := RXState.Freq_rx + RXState.Freq_step;
        UpdateDisplay;
        ParseStatus(statusC, STATUSTYPE_C);
      end else Message(COMMANDFAILED);
    end else Message(RXCANTTUNE);
  end;
  if WheelDelta < 0 then begin
    if RXState.Freq_rx-RXState.Freq_step >= MINRXF then begin
      if SendCommandC(Tune_Dn, statusC, REENABLE) then begin
        RXState.Freq_rx := RXState.Freq_rx - RXState.Freq_step;
        UpdateDisplay;
        ParseStatus(statusC, STATUSTYPE_C);
      end else Message(COMMANDFAILED);
    end else Message(RXCANTTUNE);
  end;
  UpdateDisplay;
  SpinEditUpdateOnly(SPFREQ, RXstate.Freq_RX);
  Handled := TRUE;
  In_Wheel := FALSE;
end;

// hamlib NET rigctl-more-or-less-compatible server networking code (LNET)
// Remote commands handling code is in uparse.pas and ustate.pas.
// The server code is so far incomplete, it's a little more than the bare minimum
// needed to work with WSJTX, FLDIGI, GRIG, XDX and XLOG.
// See also the comments in UParse.pas

// Accept a connection
procedure TEKD.EKD500ServerAccept(aSocket: TLSocket);

begin
  NumTCPConnections := NumTCPConnections+1;
end;

// Handle disconnect received from client
procedure TEKD.EKD500ServerDisconnect(aSocket: TLSocket);
begin
  if NumTCPConnections > 0 then NumTCPConnections := NumTCPConnections - 1;
end;

// Error handling
procedure TEKD.EKD500serverError(const amsg: string; aSocket: TLSocket);
begin
  Message(Format(ERRORNET,[amsg]));
  UpdateDisplay;
end;

// Receive data from client
procedure TEKD.EKD500serverReceive(aSocket: TLSocket);

var len: integer;

begin
  len := aSocket.GetMessage(RXLine);
  {$ifdef DEBUG_NET}
  Message('Received '+IntToStr(len)+' bytes');
  {$endif}
  if len > 0 then begin
    {$ifdef DEBUG_NET}
    Message('Net RX:'+ShowControlChars(RXLine));
    {$endif}
    ParseNetMsg(RXLine,aSocket);
  end;
end;

//
// Timers
//

// Status timer. Get Smeter every 500 ms.
procedure TEKD.TSmeterTimer(Sender: TObject);

begin
  {$IFDEF TEST_USER_INTERFACE}
  statusG := IntToStr(20+random(40)) + ' gc 1';
  exit;
  {$ENDIF}
  if Enable_Status then begin
    if In_Status or In_Command then exit;
    In_Status := TRUE;
    In_Command := TRUE;
    statusG := '';
    while not SendCommandC(Key_GC, statusG, DONT_REENABLE) do;
    while not SendCommandC(Key_E, statusC, REENABLE) do;
    {$IFDEF TEST}
    Message('status: '+statusG);
    {$ENDIF}
    In_Status := FALSE;
    In_Command := FALSE;
  end;
end;

// SMETER timer. Update SMeter every 450 ms.
procedure TEKD.TSMDispTimer(Sender: TObject);

begin
  {$IFDEF TEST_USER_INTERFACE}
  statusG := IntToStr(20+random(40)) + ' gc 1';
  {$ENDIF}

  {$IFDEF TEST}
  Message('In SM SB='+statusG);
  {$ENDIF}
  if statusG <> '' then begin
    dBm := 2*StrToInt(copy(statusG,1,2));
    SetSMeter((dBm div 10)+1);
    SMLabel.Caption := IntTostr(dbm)+' of 120 ('+CalcSValue(dbm)+')'
  end;
end;

// Timer to show net status and update main window from uparse.pas
procedure TEKD.TNETTimer(Sender: TObject);
begin
  if NumTCPConnections > 0 then
    LBNETW.Visible := TRUE
  else
    LBNETW.Visible := FALSE;

  if PanelUpdateRequired then begin
    UpdateDisplay;
    UpdateControls;
    PanelUpdateRequired := FALSE;
  end;
end;

// Things to do at startup
initialization

begin
  // Use internal Lazarus versioning
  if GetProgramVersion(ProgVersion) then begin
    Major := ProgVersion[1];
    Minor := ProgVersion[2];
    Revision := ProgVersion[3];
    Build := ProgVersion[4];
    // Make version strings
    ShortVersion := 'v'+IntTostr(Major)+'.'+IntTostr(Minor)+IntToStr(revision)+
                    ' build '+IntToStr(Build);
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end else begin
    ShowMessage(GETPROGVFAIL);
    ShortVersion := NOVERSINFO;
    LongVersion := ShortVersion+LineEnding+COPYRIGHTST;
  end;

  {$IFDEF LINUX}
  HomeDir := GetUserDir+'.EKD500Control/';
  StateDir := HomeDir+'States/';
  ChannelsDir := HomeDir+'Channels/';
  DocDir := '/usr/share/doc/ekd500control/';
  GPLv3Dir := DocDir+'GPLv3/';
  BandScansdir := Homedir+'BandScans/';
  {$ENDIF}
  {$IFDEF WINDOWS}
  // Use the executable path to discover the location of the
  // SkantiControl directory.
  HomeDir := ExtractFilePath(Application.ExeName);
  StateDir := HomeDir+'States\';
  ChannelsDir := HomeDir+'Channels\';
  DocDir := HomeDir+'doc\';
  BandScansdir := Homedir+'BandScans\';
  GPLv3Dir := HomeDir+'GPLv3\';
  Application.Initialize;
  {$ENDIF}
  if not DirectoryExists(HomeDir) then CreateDir(HomeDir);
  if not DirectoryExists(StateDir) then CreateDir(StateDir);
  if not DirectoryExists(BandsDir) then CreateDir(BandsDir);
  if not DirectoryExists(ChannelsDir) then CreateDir(ChannelsDir);
  if not DirectoryExists(BandScansDir) then CreateDir(BandScansDir);

  // Check manual and license files
  if not (FileExists(DocDir+'Manual-en.txt') and
          FileExists(GPLv3Dir+'gpl-3.0.txt')) then
    ErrMsg := DOCFILESNOTF+LineEnding+CHECKINST+LineEnding+PRGWILLHALT;

  // If installation is severely broken, notify user and close program now.
  // Moved to FormCreate since here Application is not running yet
end;

end.

